<?php
namespace WP_Improved_Settings;
/**
 * Abstract Settings API Class
 * From abstract-wc-settings-api.php
 *
 *
 * @package WP_Improved_Settings
 * @version 20190317
 */

defined( 'ABSPATH' ) || exit;

/**
 * WC_Improved_Settings_API class.
 */
if( !class_exists( 'WP_Improved_Settings\WC_Improved_Settings_API' ) ) {
    class WC_Improved_Settings_API {
        /**
         * The plugin ID. Used for option names.
         *
         * @var string
         */
        public $plugin_id = 'woocommerce_';

        /**
         * ID of the class extending the settings API. Used in option names.
         *
         * @var string
         */
        public $id = '';

        /**
         * Validation errors.
         *
         * @var array of strings
         */
        public $errors = array();

        /**
         * Setting values.
         *
         * @var array
         */
        public $settings = array();

        /**
         * Form option fields.
         *
         * @var array
         */
        public $form_fields = array();

        protected $availableUserFields = array();

        /**
         * The posted settings data. When empty, $_POST data will be used.
         *
         * @var array
         */
        protected $data = array();

        public function __construct( $plugin_id , $settingsStructure ) {
            $this->plugin_id = $plugin_id .'_';
            $this->settingsStructure = $settingsStructure;
            $this->init_form_fields();

        }

        /**
         * Initialise settings form fields.
         *
         * Add an array of fields to be displayed on the gateway's settings screen.
         *
         * @since 1.0.0
         */
        public function init_form_fields() {
            if (!$this->settingsStructure || !count($this->settingsStructure)) {
                return false;
            }
            foreach ($this->settingsStructure as $tab_id => $tab_data) {
                foreach ($tab_data['sections'] as $section_id => $section_data ) {
                    foreach ($section_data['fields'] as $field) {
                        $this->id = $tab_id;
                        $field_key = $this->get_field_key( $field['id'] );
                        $field['tab'] = $tab_id;
                        $field['section'] = $section_id;
                        $this->form_fields[$field_key] = $field;
                    }
                }
            }
        }

        function process_admin_options() {
            $post_data = $this->get_post_data();
            $current_tab = $post_data['tab'];

            if( method_exists( $this, 'before_save' ) ) {
                $this->before_save();
            }

            foreach( $this->form_fields as $field_key => $field ) {
                if($field['tab'] != $current_tab) {
                    continue;
                }
                if( isset( $post_data[$field_key] ) ) {
                    $field_value = $post_data[$field_key];

                    if( $field['type'] == 'checkbox' && $field_value == 1 ) {
                        $field_value = 'yes';
                    }
                    if( $field['type'] == 'textarea' || $field['type'] == 'text' ) {
                        $field_value = stripslashes( $field_value );
                    }
                    update_option( $field_key, $field_value );
                }
                else {
                    update_option($field_key, false);
                }
            }

            if( method_exists( $this, 'on_save' ) ) {
                $this->on_save();
            }
        }

        /**
         * Prefix key for settings.
         *
         * @param string $key Field key.
         * @return string
         */
        public function get_field_key( $key ) {
            return $this->plugin_id . $key;
        }

        public function update_option( $key, $value = '' ) {
            return update_option( $this->get_field_key( $key ), $value );
        }

        /**
         * Get option from DB.
         *
         * Gets an option from the settings API, using defaults if necessary to prevent undefined notices.
         *
         * @param string $key Option key.
         * @param mixed $empty_value Value when empty.
         * @return string The value specified for the option or a default value for the option.
         */
        public function get_option( $key, $empty_value = null ) {
            return get_option( $this->get_field_key( $key ) );
        }

        public function generate_radio_html( $key, $data ) {
            $field_key = $this->get_field_key( $key );
            $defaults = array(
                'title' => '',
                'label' => '',
                'disabled' => false,
                'class' => '',
                'css' => '',
                'type' => 'text',
                'desc_tip' => false,
                'description' => '',
                'custom_attributes' => array(),
            );

            $data = wp_parse_args( $data, $defaults );

            if ( ! $data['label'] ) {
                $data['label'] = $data['title'];
            }

            ob_start();
            ?>
            <tr valign="top">
                <th scope="row" class="titledesc">
                    <label for="<?php echo esc_attr( $field_key ); ?>"><?php echo wp_kses_post( $data['title'] ); ?> <?php echo $this->get_tooltip_html( $data ); // WPCS: XSS ok. ?></label>
                </th>
                <td class="forminp">
                    <fieldset>
                        <legend class="screen-reader-text"><span><?php echo wp_kses_post( $data['title'] ); ?></span></legend>
                        <?php
                        $stored_value = $this->get_option( $key );
                        if( !$stored_value && isset( $data['default'] ) ) {
                            $stored_value = $data['default'];
                        }

                        if( isset( $data['values'] ) ) foreach( $data['values'] as $value => $label ) :
                            //echo " V $value / S $stored_value";

                         ?>
                        <input <?php disabled( $data['disabled'], true ); ?> class="<?php echo esc_attr( $data['class'] ); ?>" type="radio" name="<?php echo esc_attr( $field_key ); ?>" id="<?php echo esc_attr( $field_key ) . '_' . sanitize_title( $value );?>" style="<?php echo esc_attr( $data['css'] ); ?>" value="<?php echo $value ?>" <?php checked( $stored_value, $value ); ?> <?php echo $this->get_custom_attribute_html( $data ); // WPCS: XSS ok. ?> />
                        <label for="<?php echo esc_attr( $field_key ); ?>"><?php echo wp_kses_post( $label ); ?></label>
                        <br />
                    <?php endforeach; ?>
                        <?php echo $this->get_description_html( $data ); // WPCS: XSS ok. ?>
                    </fieldset>
                </td>
            </tr>
            <?php

            return ob_get_clean();
        }

        public function generate_raw_html( $key, $data ) {
            $field_key = $this->get_field_key( $key );
            $defaults = array(
                'title' => '',
                'label' => '',
                'disabled' => false,
                'class' => '',
                'css' => '',
                'type' => 'text',
                'desc_tip' => false,
                'description' => '',
                'custom_attributes' => array(),
            );

            $data = wp_parse_args( $data, $defaults );

            ob_start();
            ?>
            <tr valign="top">
                <th scope="row" class="titledesc">
                    <label for="<?php echo esc_attr( $field_key ); ?>"><?php echo wp_kses_post( $data['title'] ); ?> <?php echo $this->get_tooltip_html( $data ); // WPCS: XSS ok. ?></label>
                </th>
                <td class="forminp">
                    <fieldset>
                        <legend class="screen-reader-text"><span><?php echo wp_kses_post( $data['title'] ); ?></span></legend>
                        <?php
                        $stored_value = $this->get_option( $key );
                        if( !$stored_value && isset( $data['default'] ) ) {
                            $stored_value = $data['default'];
                        }

                        echo $data['raw_html']; ?>
                        <br/>
                        <?php echo $this->get_description_html( $data ); // WPCS: XSS ok. ?>
                    </fieldset>
                </td>
            </tr>
            <?php

            return ob_get_clean();
        }


        /**
         * Get the form fields after they are initialized.
         *
         * @return array of options
         */
        public function get_form_fields() {
            return apply_filters( 'woocommerce_settings_api_form_fields_' . $this->id, array_map( array( $this, 'set_defaults' ), $this->form_fields ) );
        }

        /**
         * Set default required properties for each field.
         *
         * @param array $field Setting field array.
         * @return array
         */
        protected function set_defaults( $field ) {
            if ( ! isset( $field['default'] ) ) {
                $field['default'] = '';
            }
            return $field;
        }

        /**
         * Output the admin options table.
         */
        public function admin_options() {
            echo '<table class="form-table">' . $this->generate_settings_html( $this->get_form_fields(), false ) . '</table>'; // WPCS: XSS ok.
        }

        /**
         * Return the name of the option in the WP DB.
         *
         * @since 2.6.0
         * @return string
         */
        public function get_option_key() {
            return $this->plugin_id . $this->id . '_settings';
        }

        /**
         * Get a fields type. Defaults to "text" if not set.
         *
         * @param array $field Field key.
         * @return string
         */
        public function get_field_type( $field ) {
            return empty( $field['type'] ) ? 'text' : $field['type'];
        }

        /**
         * Get a fields default value. Defaults to "" if not set.
         *
         * @param array $field Field key.
         * @return string
         */
        public function get_field_default( $field ) {
            return empty( $field['default'] ) ? '' : $field['default'];
        }

        /**
         * Get a field's posted and validated value.
         *
         * @param string $key Field key.
         * @param array $field Field array.
         * @param array $post_data Posted data.
         * @return string
         */
        public function get_field_value( $key, $field, $post_data = array() ) {
            $type = $this->get_field_type( $field );
            $field_key = $this->get_field_key( $key );
            $post_data = empty( $post_data ) ? $_POST : $post_data; // WPCS: CSRF ok, input var ok.
            $value = isset( $post_data[ $field_key ] ) ? $post_data[ $field_key ] : null;

            if ( isset( $field['sanitize_callback'] ) && is_callable( $field['sanitize_callback'] ) ) {
                return call_user_func( $field['sanitize_callback'], $value );
            }

            // Look for a validate_FIELDID_field method for special handling.
            if ( is_callable( array( $this, 'validate_' . $key . '_field' ) ) ) {
                return $this->{'validate_' . $key . '_field'}( $key, $value );
            }

            // Look for a validate_FIELDTYPE_field method.
            if ( is_callable( array( $this, 'validate_' . $type . '_field' ) ) ) {
                return $this->{'validate_' . $type . '_field'}( $key, $value );
            }

            // Fallback to text.
            return $this->validate_text_field( $key, $value );
        }

        /**
         * Sets the POSTed data. This method can be used to set specific data, instead of taking it from the $_POST array.
         *
         * @param array $data Posted data.
         */
        public function set_post_data( $data = array() ) {
            $this->data = $data;
        }

        /**
         * Returns the POSTed data, to be used to save the settings.
         *
         * @return array
         */
        public function get_post_data() {
            if ( ! empty( $this->data ) && is_array( $this->data ) ) {
                return $this->data;
            }
            return $_POST; // WPCS: CSRF ok, input var ok.
        }

        /**
         * Add an error message for display in admin on save.
         *
         * @param string $error Error message.
         */
        public function add_error( $error ) {
            $this->errors[] = $error;
        }

        /**
         * Get admin error messages.
         */
        public function get_errors() {
            return $this->errors;
        }

        /**
         * Display admin error messages.
         */
        public function display_errors() {
            if ( $this->get_errors() ) {
                echo '<div id="woocommerce_errors" class="error notice is-dismissible">';
                foreach ( $this->get_errors() as $error ) {
                    echo '<p>' . wp_kses_post( $error ) . '</p>';
                }
                echo '</div>';
            }
        }

        /**
         * Initialise Settings.
         *
         * Store all settings in a single database entry
         * and make sure the $settings array is either the default
         * or the settings stored in the database.
         *
         * @since 1.0.0
         * @uses get_option(), add_option()
         */
        public function init_settings() {
            $this->settings = get_option( $this->get_option_key(), null );

            // If there are no settings defined, use defaults.
            if ( ! is_array( $this->settings ) ) {
                $form_fields = $this->get_form_fields();
                $this->settings = array_merge( array_fill_keys( array_keys( $form_fields ), '' ), wp_list_pluck( $form_fields, 'default' ) );
            }
        }

        /**
         * Generate Settings HTML.
         *
         * Generate the HTML for the fields on the "settings" screen.
         *
         * @param array $form_fields ( default: array() ) Array of form fields.
         * @param bool $echo Echo or return.
         * @return string the html for the settings
         * @since 1.0.0
         * @uses method_exists()
         */
        public function generate_settings_html( $form_fields = array(), $echo = true ) {
            $html = '';
            foreach ( $form_fields as $k => $v ) {
                $type = $this->get_field_type( $v );

                if ( method_exists( $this, 'generate_' . $type . '_html' ) ) {
                    $html .= $this->{'generate_' . $type . '_html'}( $k, $v );
                } else {
                    $html .= $this->generate_text_html( $k, $v );
                }
            }

            if ( $echo ) {
                echo $html; // WPCS: XSS ok.
            } else {
                return $html;
            }
        }

        /**
         * Get HTML for tooltips.
         *
         * @param array $data Data for the tooltip.
         * @return string
         */
        public function get_tooltip_html( $data ) {
            if ( true === $data['desc_tip'] ) {
                $tip = $data['description'];
            } elseif ( ! empty( $data['desc_tip'] ) ) {
                $tip = $data['desc_tip'];
            } else {
                $tip = '';
            }

            return $tip ? wc_help_tip( $tip, true ) : '';
        }

        /**
         * Get HTML for descriptions.
         *
         * @param array $data Data for the description.
         * @return string
         */
        public function get_description_html( $data ) {
            if ( true === $data['desc_tip'] ) {
                $description = '';
            } elseif ( ! empty( $data['desc_tip'] ) ) {
                $description = $data['description'];
            } elseif ( ! empty( $data['description'] ) ) {
                $description = $data['description'];
            } else {
                $description = '';
            }

            return $description ? '<p class="description">' . wp_kses_post( $description ) . '</p>' . "\n" : '';
        }

        /**
         * Get custom attributes.
         *
         * @param array $data Field data.
         * @return string
         */
        public function get_custom_attribute_html( $data ) {
            $custom_attributes = array();

            if ( ! empty( $data['custom_attributes'] ) && is_array( $data['custom_attributes'] ) ) {
                foreach ( $data['custom_attributes'] as $attribute => $attribute_value ) {
                    $custom_attributes[] = esc_attr( $attribute ) . '="' . esc_attr( $attribute_value ) . '"';
                }
            }

            return implode( ' ', $custom_attributes );
        }

        /**
         * Generate Text Input HTML.
         *
         * @param string $key Field key.
         * @param array $data Field data.
         * @since 1.0.0
         * @return string
         */
        public function generate_text_html( $key, $data ) {
            $field_key = $this->get_field_key( $key );
            $defaults = array(
                'title' => '',
                'disabled' => false,
                'class' => '',
                'css' => '',
                'placeholder' => '',
                'type' => 'text',
                'desc_tip' => false,
                'description' => '',
                'custom_attributes' => array(),
            );

            $data = wp_parse_args( $data, $defaults );

            ob_start();
            ?>
            <tr valign="top">
                <th scope="row" class="titledesc">
                    <label for="<?php echo esc_attr( $field_key ); ?>"><?php echo wp_kses_post( $data['title'] ); ?> <?php echo $this->get_tooltip_html( $data ); // WPCS: XSS ok. ?></label>
                </th>
                <td class="forminp">
                    <fieldset>
                        <legend class="screen-reader-text"><span><?php echo wp_kses_post( $data['title'] ); ?></span></legend>
                        <input class="input-text regular-input <?php echo esc_attr( $data['class'] ); ?>" type="<?php echo esc_attr( $data['type'] ); ?>" name="<?php echo esc_attr( $field_key ); ?>" id="<?php echo esc_attr( $field_key ); ?>" style="<?php echo esc_attr( $data['css'] ); ?>" value="<?php echo esc_attr( $this->get_option( $key ) ); ?>" placeholder="<?php echo esc_attr( $data['placeholder'] ); ?>" <?php disabled( $data['disabled'], true ); ?> <?php echo $this->get_custom_attribute_html( $data ); // WPCS: XSS ok. ?> />
                        <?php echo $this->get_description_html( $data ); // WPCS: XSS ok. ?>
                    </fieldset>
                </td>
            </tr>
            <?php

            return ob_get_clean();
        }

        /**
         * Generate Price Input HTML.
         *
         * @param string $key Field key.
         * @param array $data Field data.
         * @since 1.0.0
         * @return string
         */
        public function generate_price_html( $key, $data ) {
            $field_key = $this->get_field_key( $key );
            $defaults = array(
                'title' => '',
                'disabled' => false,
                'class' => '',
                'css' => '',
                'placeholder' => '',
                'type' => 'text',
                'desc_tip' => false,
                'description' => '',
                'custom_attributes' => array(),
            );

            $data = wp_parse_args( $data, $defaults );

            ob_start();
            ?>
            <tr valign="top">
                <th scope="row" class="titledesc">
                    <label for="<?php echo esc_attr( $field_key ); ?>"><?php echo wp_kses_post( $data['title'] ); ?> <?php echo $this->get_tooltip_html( $data ); // WPCS: XSS ok. ?></label>
                </th>
                <td class="forminp">
                    <fieldset>
                        <legend class="screen-reader-text"><span><?php echo wp_kses_post( $data['title'] ); ?></span></legend>
                        <input class="wc_input_price input-text regular-input <?php echo esc_attr( $data['class'] ); ?>" type="text" name="<?php echo esc_attr( $field_key ); ?>" id="<?php echo esc_attr( $field_key ); ?>" style="<?php echo esc_attr( $data['css'] ); ?>" value="<?php echo esc_attr( wc_format_localized_price( $this->get_option( $key ) ) ); ?>" placeholder="<?php echo esc_attr( $data['placeholder'] ); ?>" <?php disabled( $data['disabled'], true ); ?> <?php echo $this->get_custom_attribute_html( $data ); // WPCS: XSS ok. ?> />
                        <?php echo $this->get_description_html( $data ); // WPCS: XSS ok. ?>
                    </fieldset>
                </td>
            </tr>
            <?php

            return ob_get_clean();
        }

        /**
         * Generate Decimal Input HTML.
         *
         * @param string $key Field key.
         * @param array $data Field data.
         * @since 1.0.0
         * @return string
         */
        public function generate_decimal_html( $key, $data ) {
            $field_key = $this->get_field_key( $key );
            $defaults = array(
                'title' => '',
                'disabled' => false,
                'class' => '',
                'css' => '',
                'placeholder' => '',
                'type' => 'text',
                'desc_tip' => false,
                'description' => '',
                'custom_attributes' => array(),
            );

            $data = wp_parse_args( $data, $defaults );

            ob_start();
            ?>
            <tr valign="top">
                <th scope="row" class="titledesc">
                    <label for="<?php echo esc_attr( $field_key ); ?>"><?php echo wp_kses_post( $data['title'] ); ?> <?php echo $this->get_tooltip_html( $data ); // WPCS: XSS ok. ?></label>
                </th>
                <td class="forminp">
                    <fieldset>
                        <legend class="screen-reader-text"><span><?php echo wp_kses_post( $data['title'] ); ?></span></legend>
                        <input class="wc_input_decimal input-text regular-input <?php echo esc_attr( $data['class'] ); ?>" type="text" name="<?php echo esc_attr( $field_key ); ?>" id="<?php echo esc_attr( $field_key ); ?>" style="<?php echo esc_attr( $data['css'] ); ?>" value="<?php echo esc_attr( wc_format_localized_decimal( $this->get_option( $key ) ) ); ?>" placeholder="<?php echo esc_attr( $data['placeholder'] ); ?>" <?php disabled( $data['disabled'], true ); ?> <?php echo $this->get_custom_attribute_html( $data ); // WPCS: XSS ok. ?> />
                        <?php echo $this->get_description_html( $data ); // WPCS: XSS ok. ?>
                    </fieldset>
                </td>
            </tr>
            <?php

            return ob_get_clean();
        }

        /**
         * Generate Password Input HTML.
         *
         * @param string $key Field key.
         * @param array $data Field data.
         * @since 1.0.0
         * @return string
         */
        public function generate_password_html( $key, $data ) {
            $data['type'] = 'password';
            return $this->generate_text_html( $key, $data );
        }

        /**
         * Generate Color Picker Input HTML.
         *
         * @param string $key Field key.
         * @param array $data Field data.
         * @since 1.0.0
         * @return string
         */
        public function generate_color_html( $key, $data ) {
            $field_key = $this->get_field_key( $key );
            $defaults = array(
                'title' => '',
                'disabled' => false,
                'class' => '',
                'css' => '',
                'placeholder' => '',
                'desc_tip' => false,
                'description' => '',
                'custom_attributes' => array(),
            );

            $data = wp_parse_args( $data, $defaults );

            ob_start();
            ?>
            <tr valign="top">
                <th scope="row" class="titledesc">
                    <label for="<?php echo esc_attr( $field_key ); ?>"><?php echo wp_kses_post( $data['title'] ); ?> <?php echo $this->get_tooltip_html( $data ); // WPCS: XSS ok. ?></label>
                </th>
                <td class="forminp">
                    <fieldset>
                        <legend class="screen-reader-text"><span><?php echo wp_kses_post( $data['title'] ); ?></span></legend>
                        <span class="colorpickpreview" style="background:<?php echo esc_attr( $this->get_option( $key ) ); ?>;">&nbsp;</span>
                        <input class="colorpick <?php echo esc_attr( $data['class'] ); ?>" type="text" name="<?php echo esc_attr( $field_key ); ?>" id="<?php echo esc_attr( $field_key ); ?>" style="<?php echo esc_attr( $data['css'] ); ?>" value="<?php echo esc_attr( $this->get_option( $key ) ); ?>" placeholder="<?php echo esc_attr( $data['placeholder'] ); ?>" <?php disabled( $data['disabled'], true ); ?> <?php echo $this->get_custom_attribute_html( $data ); // WPCS: XSS ok. ?> />
                        <div id="colorPickerDiv_<?php echo esc_attr( $field_key ); ?>" class="colorpickdiv" style="z-index: 100; background: #eee; border: 1px solid #ccc; position: absolute; display: none;"></div>
                        <?php echo $this->get_description_html( $data ); // WPCS: XSS ok. ?>
                    </fieldset>
                </td>
            </tr>
            <?php

            return ob_get_clean();
        }

        /**
         * Generate Textarea HTML.
         *
         * @param string $key Field key.
         * @param array $data Field data.
         * @since 1.0.0
         * @return string
         */
        public function generate_textarea_html( $key, $data ) {
            $field_key = $this->get_field_key( $key );
            $defaults = array(
                'title' => '',
                'disabled' => false,
                'class' => '',
                'css' => '',
                'placeholder' => '',
                'type' => 'text',
                'desc_tip' => false,
                'description' => '',
                'custom_attributes' => array(),
            );

            $data = wp_parse_args( $data, $defaults );

            ob_start();
            ?>
            <tr valign="top">
                <th scope="row" class="titledesc">
                    <label for="<?php echo esc_attr( $field_key ); ?>"><?php echo wp_kses_post( $data['title'] ); ?> <?php echo $this->get_tooltip_html( $data ); // WPCS: XSS ok. ?></label>
                </th>
                <td class="forminp">
                    <fieldset>
                        <legend class="screen-reader-text"><span><?php echo wp_kses_post( $data['title'] ); ?></span></legend>
                        <textarea rows="3" cols="20" class="input-text wide-input <?php echo esc_attr( $data['class'] ); ?>" type="<?php echo esc_attr( $data['type'] ); ?>" name="<?php echo esc_attr( $field_key ); ?>" id="<?php echo esc_attr( $field_key ); ?>" style="<?php echo esc_attr( $data['css'] ); ?>" placeholder="<?php echo esc_attr( $data['placeholder'] ); ?>" <?php disabled( $data['disabled'], true ); ?> <?php echo $this->get_custom_attribute_html( $data ); // WPCS: XSS ok. ?>><?php echo esc_textarea( $this->get_option( $key ) ); ?></textarea>
                        <?php echo $this->get_description_html( $data ); // WPCS: XSS ok. ?>
                    </fieldset>
                </td>
            </tr>
            <?php

            return ob_get_clean();
        }

        /**
         * Generate Checkbox HTML.
         *
         * @param string $key Field key.
         * @param array $data Field data.
         * @since 1.0.0
         * @return string
         */
        public function generate_checkbox_html( $key, $data ) {
            $field_key = $this->get_field_key( $key );
            $defaults = array(
                'title' => '',
                'label' => '',
                'disabled' => false,
                'class' => '',
                'css' => '',
                'type' => 'text',
                'desc_tip' => false,
                'description' => '',
                'custom_attributes' => array(),
            );

            $data = wp_parse_args( $data, $defaults );

            if ( ! $data['label'] ) {
                $data['label'] = $data['title'];
            }

            ob_start();
            ?>
            <tr valign="top">
                <th scope="row" class="titledesc">
                    <label for="<?php echo esc_attr( $field_key ); ?>"><?php echo wp_kses_post( $data['title'] ); ?> <?php echo $this->get_tooltip_html( $data ); // WPCS: XSS ok. ?></label>
                </th>
                <td class="forminp">
                    <fieldset>
                        <legend class="screen-reader-text"><span><?php echo wp_kses_post( $data['title'] ); ?></span></legend>
                        <label for="<?php echo esc_attr( $field_key ); ?>">
                        <input <?php disabled( $data['disabled'], true ); ?> class="<?php echo esc_attr( $data['class'] ); ?>" type="checkbox" name="<?php echo esc_attr( $field_key ); ?>" id="<?php echo esc_attr( $field_key ); ?>" style="<?php echo esc_attr( $data['css'] ); ?>" value="1" <?php checked( $this->get_option( $key ), 'yes' ); ?> <?php echo $this->get_custom_attribute_html( $data ); // WPCS: XSS ok. ?> /> <?php echo wp_kses_post( $data['label'] ); ?></label><br/>
                        <?php echo $this->get_description_html( $data ); // WPCS: XSS ok. ?>
                    </fieldset>
                </td>
            </tr>
            <?php

            return ob_get_clean();
        }

        /**
         * Generate Select HTML.
         *
         * @param string $key Field key.
         * @param array $data Field data.
         * @since 1.0.0
         * @return string
         */
        public function generate_select_html( $key, $data ) {
            $data['options'] = count($data['options']) > 0 ? $data['options'] : $this->generate_options_select($key);
            $field_key = $this->get_field_key( $key );
            $defaults = array(
                'title' => '',
                'disabled' => false,
                'class' => '',
                'css' => '',
                'placeholder' => '',
                'type' => 'text',
                'desc_tip' => false,
                'description' => '',
                'custom_attributes' => array(),
                'options' => array(),
            );

            $data = wp_parse_args( $data, $defaults );

            ob_start();
            ?>
            <tr valign="top">
                <th scope="row" class="titledesc">
                    <label for="<?php echo esc_attr( $field_key ); ?>"><?php echo wp_kses_post( $data['title'] ); ?> <?php echo $this->get_tooltip_html( $data ); // WPCS: XSS ok. ?></label>
                </th>
                <td class="forminp">
                    <fieldset>
                        <legend class="screen-reader-text"><span><?php echo wp_kses_post( $data['title'] ); ?></span></legend>
                        <select class="select <?php echo esc_attr( $data['class'] ); ?>" name="<?php echo esc_attr( $field_key ); ?>" id="<?php echo esc_attr( $field_key ); ?>" style="<?php echo esc_attr( $data['css'] ); ?>" <?php disabled( $data['disabled'], true ); ?> <?php echo $this->get_custom_attribute_html( $data ); // WPCS: XSS ok. ?>>
                            <?php foreach ( ( array ) $data['options'] as $option_key => $option_value ) : ?>
                                <option value="<?php echo esc_attr( $option_key ); ?>" <?php selected( ( string ) $option_key, esc_attr( $this->get_option( $key ) ) ); ?>><?php echo esc_attr( $option_value ); ?></option>
                            <?php endforeach; ?>
                        </select>
                        <?php echo $this->get_description_html( $data ); // WPCS: XSS ok. ?>
                    </fieldset>
                </td>
            </tr>
            <?php

            return ob_get_clean();
        }

        public function generate_options_select($key) {
            $options = array();
            switch ($key) {
                case 'shop_contact_url':
                    $results = get_pages();
                    $pages = array();
                    if($results) foreach($results as $result) {
                        $pages[$result->ID] = $result->post_title;
                    }
                    $options = $pages;
                    break;
                case 'email_image_format':
                    $sizes = \ShopiMindConfiguration::getWordPressImagesSize();
                    $available_image_sizes = array();
                    if( $sizes ) foreach( $sizes as $size => $dimensions ) {
                        $available_image_sizes[$size] = sprintf(
                            __( '%s ( %dx%d )', 'shopimind' ),
                            $size,
                            $dimensions['width'],
                            $dimensions['height']
                        );
                    }
                    $options = $available_image_sizes;
                    break;
                case 'birthday_field':
                case 'gender_field':
                case 'newsletter_field':
                    if(count($this->availableUserFields) === 0) {
                        $this->availableUserFields = $this->getAvailableUserFields();
                    }
                    $options = array('none' => __( 'None', 'shopimind' )) + $this->availableUserFields;
                    break;
                case 'manufacturer_field':
                    $options = array('none'	=> __( 'None', 'shopimind' )) + $this->getAvailableProductFields();
                    break;
                case 'manufacturer_taxonomy':
                    $taxonomies = get_taxonomies( array( 'object_type' => array( 'product' )), 'objects' );
                    $available_taxonomies = array( 'none'	=> __( 'None', 'shopimind' ));
                    if( $taxonomies ) foreach( $taxonomies as $taxonomy => $tax_data ) {
                        $available_taxonomies[$taxonomy] = $tax_data->labels->name;
                    }
                    $options = $available_taxonomies;
                    break;
                case 'ml_table_name':
                    $wp_tables = \ShopiMindConfiguration::getTableNames();
                    $options = array('none'    => __( 'None', 'shopimind' )) + $wp_tables;
                    break;
                case (bool)preg_match('/^ml_field_/', $key):
                    $ml_sync_fields = \ShopiMindConfiguration::getMLTableFields();
                    if ($ml_sync_fields && is_array($ml_sync_fields)) {
                        array_unshift($ml_sync_fields, '-');
                        $ml_sync_fields = array_combine($ml_sync_fields, $ml_sync_fields);
                        $options = $ml_sync_fields;
                    }
                    break;
            }

            return $options;
        }

        function getAvailableUserFields() {
            global $wpdb;
            $ignored_meta_keys = array(
                'nickname', 'first_name', 'last_name', 'description', 'rich_editing', 'syntax_highlighting', 'comment_shortcuts', 'admin_color', 'use_ssl', 'show_admin_bar_front',
                'locale', 'wp_capabilities', 'wp_user_level', 'dismissed_wp_pointers', 'show_welcome_panel', 'session_tokens',
                'wp_user-settings', 'wp_user-settings-time', 'wp_dashboard_quick_press_last_post_id', 'community-events-location', 'wc_last_active',
                'dismissed_no_secure_connection_notice', 'closedpostboxes_dashboard', 'metaboxhidden_dashboard', 'meta-box-order_dashboard', 'closedpostboxes_listing',
                'metaboxhidden_listing', 'last_update',
                'billing_first_name', 'billing_last_name', 'billing_address_1', 'billing_city', 'billing_postcode', 'billing_country', 'billing_email', 'billing_phone',
                'shipping_first_name', 'shipping_last_name', 'shipping_address_1', 'shipping_city', 'shipping_postcode', 'shipping_country', 'shipping_method',
                'billing_company', 'billing_address_2', 'billing_state',
                'shipping_company', 'shipping_address_2', 'shipping_state', 'dismissed_wootenberg_notice',
                'health-check','jetpack_tracks_anon_id',
                'source_domain', 'primary_blog' );
            $sql = "SELECT DISTINCT( meta_key ) FROM $wpdb->usermeta WHERE meta_key NOT IN ( '" . implode( "', '", $ignored_meta_keys ) ."' )";
            $results = $wpdb->get_col( $sql );

            $ignore_also = array('_capabilities','user_level','user-settings','user-settings-time','dashboard_quick_press_last_post_id');

            $meta_keys = array();
            if( $results ) foreach( $results as $meta_key ) {
                if( substr( $meta_key, 0, 1 ) == '_' ) {
                    continue;
                }
                foreach( $ignore_also as $string ) {
                    if( substr( $meta_key, -1 * strlen($string) ) === $string ) {
                        continue 2;
                    }
                }

                $meta_keys[$meta_key] = $meta_key;
            }

            return $meta_keys;
        }

        function getAvailableProductFields() {
            global $wpdb;
            $ignored_meta_keys = array();

            $sql = "SELECT DISTINCT( meta_key ) FROM $wpdb->postmeta WHERE post_id IN ( SELECT ID FROM $wpdb->posts WHERE post_type = 'product' ) AND SUBSTR(meta_key,1,1) <> '_' ";

            if( count( $ignored_meta_keys )) {
                $sql .= " AND meta_key NOT IN ( '" . implode( "', '", $ignored_meta_keys ) ."' )";
            }

            $results = $wpdb->get_col( $sql );
            $meta_keys = array();
            if( $results ) foreach( $results as $meta_key ) {
                $meta_keys[$meta_key] = $meta_key;
            }

            return $meta_keys;
        }

        /**
         * Generate Multiselect HTML.
         *
         * @param string $key Field key.
         * @param array $data Field data.
         * @since 1.0.0
         * @return string
         */
        public function generate_multiselect_html( $key, $data ) {
            $field_key = $this->get_field_key( $key );
            $defaults = array(
                'title' => '',
                'disabled' => false,
                'class' => '',
                'css' => '',
                'placeholder' => '',
                'type' => 'text',
                'desc_tip' => false,
                'description' => '',
                'custom_attributes' => array(),
                'select_buttons' => false,
                'options' => array(),
            );

            $data = wp_parse_args( $data, $defaults );
            $value = ( array ) $this->get_option( $key, array() );

            ob_start();
            ?>
            <tr valign="top">
                <th scope="row" class="titledesc">
                    <label for="<?php echo esc_attr( $field_key ); ?>"><?php echo wp_kses_post( $data['title'] ); ?> <?php echo $this->get_tooltip_html( $data ); // WPCS: XSS ok. ?></label>
                </th>
                <td class="forminp">
                    <fieldset>
                        <legend class="screen-reader-text"><span><?php echo wp_kses_post( $data['title'] ); ?></span></legend>
                        <select multiple="multiple" class="multiselect <?php echo esc_attr( $data['class'] ); ?>" name="<?php echo esc_attr( $field_key ); ?>[]" id="<?php echo esc_attr( $field_key ); ?>" style="<?php echo esc_attr( $data['css'] ); ?>" <?php disabled( $data['disabled'], true ); ?> <?php echo $this->get_custom_attribute_html( $data ); // WPCS: XSS ok. ?>>
                            <?php foreach ( ( array ) $data['options'] as $option_key => $option_value ) : ?>
                                <?php if ( is_array( $option_value ) ) : ?>
                                    <optgroup label="<?php echo esc_attr( $option_key ); ?>">
                                        <?php foreach ( $option_value as $option_key_inner => $option_value_inner ) : ?>
                                            <option value="<?php echo esc_attr( $option_key_inner ); ?>" <?php selected( in_array( ( string ) $option_key_inner, $value, true ), true ); ?>><?php echo esc_attr( $option_value_inner ); ?></option>
                                        <?php endforeach; ?>
                                    </optgroup>
                                <?php else : ?>
                                    <option value="<?php echo esc_attr( $option_key ); ?>" <?php selected( in_array( ( string ) $option_key, $value, true ), true ); ?>><?php echo esc_attr( $option_value ); ?></option>
                                <?php endif; ?>
                            <?php endforeach; ?>
                        </select>
                        <?php echo $this->get_description_html( $data ); // WPCS: XSS ok. ?>
                        <?php if ( $data['select_buttons'] ) : ?>
                            <br/><a class="select_all button" href="#"><?php esc_html_e( 'Select all', 'woocommerce' ); ?></a> <a class="select_none button" href="#"><?php esc_html_e( 'Select none', 'woocommerce' ); ?></a>
                        <?php endif; ?>
                    </fieldset>
                </td>
            </tr>
            <?php

            return ob_get_clean();
        }

        /**
         * Generate Title HTML.
         *
         * @param string $key Field key.
         * @param array $data Field data.
         * @since 1.0.0
         * @return string
         */
        public function generate_title_html( $key, $data ) {
            $field_key = $this->get_field_key( $key );
            $defaults = array(
                'title' => '',
                'class' => '',
            );

            $data = wp_parse_args( $data, $defaults );

            ob_start();
            ?>
                </table>
                <h3 class="wc-settings-sub-title <?php echo esc_attr( $data['class'] ); ?>" id="<?php echo esc_attr( $field_key ); ?>"><?php echo wp_kses_post( $data['title'] ); ?></h3>
                <?php if ( ! empty( $data['description'] ) ) : ?>
                    <p><?php echo wp_kses_post( $data['description'] ); ?></p>
                <?php endif; ?>
                <table class="form-table">
            <?php

            return ob_get_clean();
        }

        /**
         * Validate Text Field.
         *
         * Make sure the data is escaped correctly, etc.
         *
         * @param string $key Field key.
         * @param string $value Posted Value.
         * @return string
         */
        public function validate_text_field( $key, $value ) {
            $value = is_null( $value ) ? '' : $value;
            return wp_kses_post( trim( stripslashes( $value ) ) );
        }

        /**
         * Validate Price Field.
         *
         * Make sure the data is escaped correctly, etc.
         *
         * @param string $key Field key.
         * @param string $value Posted Value.
         * @return string
         */
        public function validate_price_field( $key, $value ) {
            $value = is_null( $value ) ? '' : $value;
            return ( '' === $value ) ? '' : wc_format_decimal( trim( stripslashes( $value ) ) );
        }

        /**
         * Validate Decimal Field.
         *
         * Make sure the data is escaped correctly, etc.
         *
         * @param string $key Field key.
         * @param string $value Posted Value.
         * @return string
         */
        public function validate_decimal_field( $key, $value ) {
            $value = is_null( $value ) ? '' : $value;
            return ( '' === $value ) ? '' : wc_format_decimal( trim( stripslashes( $value ) ) );
        }

        /**
         * Validate Password Field. No input sanitization is used to avoid corrupting passwords.
         *
         * @param string $key Field key.
         * @param string $value Posted Value.
         * @return string
         */
        public function validate_password_field( $key, $value ) {
            $value = is_null( $value ) ? '' : $value;
            return trim( stripslashes( $value ) );
        }

        /**
         * Validate Textarea Field.
         *
         * @param string $key Field key.
         * @param string $value Posted Value.
         * @return string
         */
        public function validate_textarea_field( $key, $value ) {
            $value = is_null( $value ) ? '' : $value;
            return wp_kses( trim( stripslashes( $value ) ),
                array_merge(
                    array(
                        'iframe' => array(
                            'src' => true,
                            'style' => true,
                            'id' => true,
                            'class' => true,
                        ),
                    ),
                    wp_kses_allowed_html( 'post' )
                )
            );
        }

        /**
         * Validate Checkbox Field.
         *
         * If not set, return "no", otherwise return "yes".
         *
         * @param string $key Field key.
         * @param string $value Posted Value.
         * @return string
         */
        public function validate_checkbox_field( $key, $value ) {
            return ! is_null( $value ) ? 'yes' : 'no';
        }

        /**
         * Validate Select Field.
         *
         * @param string $key Field key.
         * @param string $value Posted Value.
         * @return string
         */
        public function validate_select_field( $key, $value ) {
            $value = is_null( $value ) ? '' : $value;
            return wc_clean( stripslashes( $value ) );
        }

        /**
         * Validate Multiselect Field.
         *
         * @param string $key Field key.
         * @param string $value Posted Value.
         * @return string|array
         */
        public function validate_multiselect_field( $key, $value ) {
            return is_array( $value ) ? array_map( 'wc_clean', array_map( 'stripslashes', $value ) ) : '';
        }

        /**
         * Validate the data on the "Settings" form.
         *
         * @deprecated 2.6.0 No longer used.
         * @param array $form_fields Array of fields.
         */
        public function validate_settings_fields( $form_fields = array() ) {
            wc_deprecated_function( 'validate_settings_fields', '2.6' );
        }

        /**
         * Format settings if needed.
         *
         * @deprecated 2.6.0 Unused.
         * @param array $value Value to format.
         * @return array
         */
        public function format_settings( $value ) {
            wc_deprecated_function( 'format_settings', '2.6' );
            return $value;
        }
    }
}
