<?php
namespace ShopiMind;

if( !class_exists( 'APIServer' ) ) {
    require_once( trailingslashit( SHOPIMIND_PATH ) . 'api/apiserver.class.php' );
}
class ShopiMindClient_Callback extends APIServer {
    protected $_methodsVersion = '1';
    protected $_methodsLastModification = '20190213';

    protected $_methods 	= array(
        'generateKeysAccess'	=> array( 'shopIdShop' ),
        'sayHello'				=> array( 'shopIdShop' ),
        'resetModule'			=> array( 'shopIdShop' ),
        'disableModule'			=> array( 'shopIdShop' ),
        'getLangs'				=> array( 'shopIdShop' ),
        'deleteUnusedVouchers'	=> array( 'shopIdShop' ),
        'dailyUpdate'	=> array( 'shopIdShop' ),
        'generateVouchers'		=> array( 'shopIdShop', 'voucherInfos', 'voucherEmails' ),
        'syncVouchers'			=> array( 'shopIdShop', 'start', 'limit', 'lastUpdate', 'idVoucher', 'justCount' ),
        'getCustomerGroups'		=> array( 'shopIdShop' ),
        'syncCustomersGroups'	=> array( 'shopIdShop', 'start', 'limit', 'lastUpdate', 'idGroup', 'justCount' ),
        'syncCustomers'			=> array( 'shopIdShop', 'start', 'limit', 'lastUpdate', 'idCustomer', 'justCount' ),
        'createCustomer'		=> array( 'shopIdShop', 'customer' ),
        'getUser'				=> array( 'shopIdShop', 'idUser' ),
        'getCarriers'			=> array( 'shopIdShop' ),
        'syncCarriers'			=> array( 'shopIdShop', 'start', 'limit', 'justCount' ),
        'syncManufacturers'		=> array( 'shopIdShop', 'start', 'limit', 'lastUpdate', 'idManufacturer', 'justCount' ),
        'syncProductsCategories'=> array( 'shopIdShop', 'start', 'limit', 'lastUpdate', 'idCategory', 'justCount' ),
        'syncProducts'			=> array( 'shopIdShop', 'start', 'limit', 'lastUpdate', 'idProduct', 'justCount' ),
        'orderStatus'			=> array( 'shopIdShop' ),
        'syncOrdersStatus'		=> array( 'shopIdShop', 'start', 'limit', 'lastUpdate', 'idStatus', 'justCount' ),
        'syncOrders'			=> array( 'shopIdShop', 'start', 'limit', 'lastUpdate', 'idOrder', 'justCount' ),
        'getOrder'				=> array( 'shopIdShop', 'idOrder' ),
        'syncNewsletters'		=> array( 'shopIdShop', 'start', 'limit', 'lastUpdate', 'justCount' ),
        'unsubCustomer'			=> array( 'shopIdShop', 'idCustomer', 'email' ),
        'setScriptUrl'			=> array( 'shopIdShop', 'script_url' ),
    );

    public $_default_​shopIdShop = 1;

    protected $_mandatory_params = array(
        'generateVouchers'		=> array( 'voucherInfos' ),
        'syncVouchers'			=> array(), // 'id_cart_rule'
        'getCustomerGroups'		=> array(),
        'syncCustomersGroups'	=> array(),
        'syncCustomers'			=> array(),
        'createCustomer'		=> array( 'customer' ),
        'getUser'				=> array( 'idUser' ),
        'getCarriers'			=> array(),
        'syncCarriers'			=> array(),
        'syncManufacturers'		=> array(),
        'syncProductsCategories' => array(),
        'syncProducts'			=> array(),
        'orderStatus'			=> array(),
        'syncOrdersStatus'		=> array(),
        'syncOrders'			=> array(),
        'getOrder'				=> array( 'idOrder' ),
        'syncNewsletters'		=> array(),
        'unsubCustomer'			=> array(),
        'setScriptUrl'			=> array( 'script_url' ),
    );

    protected $_parameters_validation 	= array(
        'shopIdShop'	=> FILTER_VALIDATE_INT,
        'start'			=> FILTER_VALIDATE_INT,
        'lastUpdate'	=> 'timestamp',
        'limit'			=> FILTER_VALIDATE_INT,
        'idVoucher'	    => FILTER_VALIDATE_INT,
        'justCount'		=> FILTER_VALIDATE_BOOLEAN,
        'idUser'		=> FILTER_VALIDATE_INT,
        'idCustomer'	=> FILTER_VALIDATE_INT,
        'idGroup'		=> FILTER_SANITIZE_STRING,
        'idManufacturer'=> FILTER_VALIDATE_INT,
        'idCategory'	=> FILTER_VALIDATE_INT,
        'idProduct'		=> FILTER_VALIDATE_INT,
        'idOrder'		=> FILTER_VALIDATE_INT,
        'idStatus'		=> FILTER_VALIDATE_INT,
        'email'			=> FILTER_VALIDATE_EMAIL,
        'customer'	=> true,
        'voucherInfos'	=> true,
        'voucherEmails'	=> true,
        'script_url' => true,
    );

    function getShopID() {
        return intval( \get_current_blog_id() );
    }


    /** 1 PULL METHODS **/
    /** GENERAL **/
    /**
     * Undocumented
     */
    public function generateKeysAccess() {
    }

    /**
     * Validate proper installation
     */
    public function selfCheck() {
        $woocommerce = false;
        // woocommerce
        if ( class_exists( 'WooCommerce' ) ) {
            $woocommerce = true;
        }

        if( !$woocommerce ) {
            return new APIError( 'website', __( "WooCommerce n'est pas installé ou pas configuré", "shopimind" ) );
        }

        $api_user = \ShopiMindConfiguration::getAPIUser();
        $api_password = \ShopiMindConfiguration::getAPIPassword();
        if( !$api_user || !$api_password || empty( $api_user ) || empty( $api_password ) ) {
            return new APIError( 'configuration', __( "Pas d'identifiants ShopiMind connus", "shopimind" ) );
        }

        return true;
    }

    /**
     * Renvoie un tableau d'informations basique
     * @param	null
     * @return	array ( version, ecommerceVersion, moduleVersion )
     */
    public function sayHello() {
        global $woocommerce;
        $woocommerce_version = $woocommerce->version;

        $return = array(
            'version'			=> SHOPIMIND_VERSION,
            'ecommerceVersion'	=> $woocommerce_version,
            'moduleVersion'		=> $this->getModuleVersion()
        );
        return $return;
    }

    /**
     * Remet la configuration d'origine
     * @param	null
     * @return	bool
     */

    public function resetModule() {
        require_once( trailingslashit( SHOPIMIND_PATH ) . 'includes/shopimind-plugin-install.php' );
        \shopimind_remove_plugin();
        \shopimind_install_plugin();
        return true;
    }

    /**
     * Unactivate plugin
     * @param	null
     * @return	array( success => bool )
     */
    public function disableModule() {
        \shopimind_deactivate_plugin();
        return true;
    }

    /**
     * Return list of front end locales
     * @param	null
     * @return	array ( langs => array( isocode => name ) )
     */
    public function getLangs( $args ) {

        $blog_ids = $this->getBlogIds( $args);
        $langs = array();
        $blog_id = array_pop( $blog_ids );
        if( function_exists( 'switch_to_blog' ) && \get_current_blog_id() != $blog_id ) {
            \switch_to_blog( $blog_id );
        }

        $descriptions = \shopimind_languages_descriptions();
        $available_languages = \get_available_languages();

        if( $available_languages ) foreach( $available_languages as $available_language ) {
            $isocode = substr( $available_language, 0, 2 );
            $description = '';
            if( isset( $descriptions[$isocode] ) ) {

                $description = $descriptions[$isocode];
            }
            $langs[$isocode] = $description;
        }

        if( function_exists( 'restore_current_blog' ) ) {
            \restore_current_blog();
        }
        $return = compact( 'langs' );
        return \apply_filters( __METHOD__,  $return );
    }

    /** VOUCHERS **/


    public function dailyUpdate() {
        return $this->deleteUnusedVouchers();
    }
    /**
     * Supprime les coupons périmés
     * @internal appelé par dailyUpdate
     * @param	null
     * @return	array( deleteUnusedVouchers => bool )
     */
    public function deleteUnusedVouchers() {
        $date = new \DateTime();
        $date->sub(new \DateInterval('P7D'));

        $args = array(
            'posts_per_page'=> -1,
            'orderby'		=> 'title',
            'order'			=> 'asc',
            'post_type'		=> 'shop_coupon',
            'post_status'	=> 'publish',
            'fields'		=> 'ids',
            's'				=> 'SPM',
            'meta_query'	=> array(
                'relation'   => 'AND',
                array(
                    'key'     => 'date_expires',
                    'value'   => $date->format( 'U' ),
                    'compare' => '<='
                ),
                array(
                    'key'     => 'date_expires',
                    'value'   => '',
                    'compare' => '!='
                )
            ),
        );

        $expired_coupons = \get_posts( $args );

        $deleteUnusedVouchers = true;
        if( $expired_coupons ) foreach( $expired_coupons as $post_id ) {
            $WC_Coupon = new \WC_Coupon( $post_id );
            $usage_count = $WC_Coupon->get_usage_count();
            if( $usage_count == 0 ) {
                \wp_delete_post( $WC_Coupon->get_id(), true );
                $deleteUnusedVouchers = true;
            }
        }

        $return = compact( 'deleteUnusedVouchers' );
        return \apply_filters( 'ShopiMind_deleteUnusedVouchers', $return );
    }


    /**
     * Génére des coupons pour les utilisateurs listés dans $voucherEmails avec les paramètres de $voucherInfos
     * @param	array $voucherInfos 	array(
     * 		'type'				=> shipping|amount|percent
     * 		'amount'			=> float
     * 		'amountCurrency'	=> string
     * 		'minimumOrder'		=> float
     * 		'nbDayValidate'		=> integer
     * 		'dynamicPrefix'		=> string
     * 		'duplicateCode'		=> ? boolean ? true pour forcer à réutiliser un code
     *
     *
     * @param	array $voucherEmails 	liste des emails clients
     * @param 	array $voucherInfos		parametres du
     * @return	array ( vouchers => array( email => voucher_code ) )
     */
    public function generateVouchers( $args ) {
        $customer_emails = array();
        if( isset( $args['voucherEmails'] ) && $args['voucherEmails'] ) foreach( $args['voucherEmails'] as $email ) {
            if( !empty( $email['email'] )) {
                $customer_emails[] = $email['email'];
                $customer_descriptions = $email['description'];
            }
        }

        $coupon_id = false;
        $blog_ids = $this->getBlogIds( $args );

        foreach($blog_ids as $blog_id ) {
            if( function_exists( 'switch_to_blog' ) && \get_current_blog_id() != $blog_id ) {
                \switch_to_blog( $blog_id );
            }

            /* Généré le code de réduction */
            if (!isset($args['voucherInfos']['codeToGenerate'])) {
                $coupon_code = generateCouponCode($args['voucherInfos']);
            } else {
                $coupon_code = $args['voucherInfos']['codeToGenerate'];
            }
            $admin_userid = getAdminUserID();
            $coupon_data = array(
                'post_title'	=> $coupon_code,
                'post_content'	=> '',
                'post_excerpt' => $customer_descriptions ?? '',
                'post_status'	=> 'publish',
                'post_author'	=> $admin_userid,
                'post_type'		=> 'shop_coupon',
            );
            $coupon_id = \wp_insert_post( $coupon_data );

            if (\is_wp_error($coupon_id)) {
                return new API_Error( 'coupon', $coupon_id->get_error_message() );
            }

            $coupon_to_duplicate = false;
            if ($args['voucherInfos']['type'] == "duplicateCode" && $args['voucherInfos']['duplicateCode']) {
                $coupon_to_duplicate = \get_page_by_title($args['voucherInfos']['duplicateCode'], ARRAY_A, 'shop_coupon');
                if(!$coupon_to_duplicate ) {
                    return new API_Error( 'coupon', "Le coupon à dupliquer est inexistant.");
                }

                /* Récupérer toutes les post_meta du code de réduction */
                $duplicated_coupon_post_meta = \get_post_meta( $coupon_to_duplicate['ID']);

                foreach ($duplicated_coupon_post_meta as $keyPostMeta => $valuePostMeta) {
                    if (in_array($keyPostMeta, ['_edit_lock', '_edit_last', 'minimum_amount'])) {
                        continue;
                    }
                    $singleValuePostMeta = \get_post_meta($coupon_to_duplicate['ID'], $keyPostMeta, true);
                    \update_post_meta($coupon_id, $keyPostMeta, $singleValuePostMeta);
                }

                $coupon_to_duplicate = true;
            } else {
                if ($args['voucherInfos']['type'] == 'amount') {
                    \update_post_meta( $coupon_id, 'discount_type', 'fixed_cart' );
                    $amount = filter_var( $args['voucherInfos']['amount'], FILTER_VALIDATE_FLOAT );
                    \update_post_meta( $coupon_id, 'coupon_amount', $amount );
                    \update_post_meta( $coupon_id, 'free_shipping', 'no' );
                }
                elseif ($args['voucherInfos']['type'] == 'percent') {
                    \update_post_meta( $coupon_id, 'discount_type', 'percent' );
                    $amount = filter_var( $args['voucherInfos']['amount'], FILTER_VALIDATE_FLOAT );
                    \update_post_meta( $coupon_id, 'coupon_amount', $amount );
                    \update_post_meta( $coupon_id, 'free_shipping', 'no' );
                } elseif ($args['voucherInfos']['type'] == 'shipping') {
                    \update_post_meta( $coupon_id, 'discount_type', 'fixed_cart' );
                    \update_post_meta( $coupon_id, 'coupon_amount', '0' );
                    \update_post_meta( $coupon_id, 'free_shipping', 'yes' );
                }
            }

            $minimumOrder = filter_var( $args['voucherInfos']['minimumOrder'], FILTER_VALIDATE_FLOAT );
            if( $minimumOrder > 0 ) {
                \update_post_meta( $coupon_id, 'minimum_amount', $minimumOrder );
            }

            $nbDayValidate = filter_var( $args['voucherInfos']['nbDayValidate'], FILTER_VALIDATE_INT );
            if (!$nbDayValidate || empty($nbDayValidate)) {
                $nbDayValidate = \ShopiMindConfiguration::getCouponsDuration();
            }
            $expiry_date = date( 'Y-m-d H:i:s', strtotime( "+ $nbDayValidate days" ));

            \update_post_meta($coupon_id, 'date_expires', $expiry_date);

            if (!$coupon_to_duplicate) {
                \update_post_meta($coupon_id, 'usage_limit_per_user', 1);
                \update_post_meta($coupon_id, 'usage_limit', 1);
            }

            $areCouponsCombinable = \ShopiMindConfiguration::areCouponsCombinable();
            if ($areCouponsCombinable) {
                \update_post_meta( $coupon_id, 'individual_use', 'no');
            } else {
                \update_post_meta( $coupon_id, 'individual_use', 'yes');
            }

            $vouchers = array();
            if($customer_emails)  {
                $useNamedCoupons = \ShopiMindConfiguration::useNamedCoupons();
                if($useNamedCoupons)
                    \update_post_meta( $coupon_id, 'customer_email', $customer_emails );

                foreach( $customer_emails as $customer_email ) {
                    $vouchers[$customer_email] = array(
                        'voucher_number'	=> $coupon_code,
                        'voucher_date_limit'	=> $expiry_date
                    );
                }
            }
            else {
                $vouchers[$coupon_code] = 1;
            }
        }
        if( function_exists( 'restore_current_blog' ) ) {
            \restore_current_blog();
        }

        $success = true;
        $return = compact( 'success', 'vouchers' );
        return \apply_filters( 'ShopiMindAPI_generateVouchers', $return, $args );
    }

    /**
     * Renvoie une liste détaillée des coupons
     * Permet de récupérer la liste des bons de réduction présents dans la base
     * Cette méthode est utilisée pour la fonctionnalité de campagne SMS et Email
     * @param	timestamp 	$start
     * @param	integer 	$limit
     * @param	timestamp 	$lastUpdate
     * @param	integer 	$idVoucher
     * @param	bool 		$justCount
     * @return	array 		( lastUpdate => timestamp, vouchers => array( id_shop, id_voucher, code, date_from, date_to, active, id_customer_shop, date_creation, date_update, minimum_amount, minimum_amount_currency, lang, ) )
     */

    public function syncVouchers( $args ) {
        $vouchers = array();

        $defaults = array(
            'start'	=> 0,
            'limit'	=> 100,
        );
        $args = wp_parse_args( $args, $defaults );
        $coupons_args = array(
            'posts_per_page'	=> -1,
            'orderby'			=> 'title',
            'order'			=> 'asc',
            'post_type'		=> 'shop_coupon',
            'post_status'		=> 'publish',
            'fields'			=> 'ids',
            'offset'			=> $args['start'],
            'post_per_page'	=> $args['limit'],
        );

        if(isset($args['idVoucher']) && intval($args['idVoucher'])) {
            $coupons_args['include'] = array(intval($args['idVoucher']));
        }

        $lastUpdate =  false;
        if( isset( $args['lastUpdate'] ) ) {
            $lastUpdate = strtotime(  /*get_gmt_from_date*/( $args['lastUpdate'] ) );
            $coupon_args['date_query'] = array(
                'after' => $lastUpdate,
            );
        }

        $wc_coupons_id = \get_posts( $coupons_args );


        if( isset($args['justCount']) && $args['justCount'] ) {

            $count = 0;
            if( $lastUpdate ) {
                if( $wc_coupons_id ) foreach( $wc_coupons_id as $wc_coupon_id ) {
                    $WC_Coupon = new \WC_Coupon( $wc_coupon_id );
                    $timestamp = strtotime( $WC_Coupon->get_date_modified()->date_i18n( 'Y-m-d H:i:s' ) );
                    if( $timestamp > $lastUpdate ) {
                        $vouchers[] = "adding $wc_coupon_id because " . date('Y-m-d H:i:s', $timestamp) . " > " . date('Y-m-d H:i:s', $lastUpdate);
                        $count++;
                    }
                    else {
                        $vouchers[] = "skipping $wc_coupon_id because " . date('Y-m-d H:i:s', $timestamp) . " <= " . date('Y-m-d H:i:s', $lastUpdate);
                        continue;
                    }
                }

            }
            else {
                $count = count( $wc_coupons_id);
            }
            $vouchers = compact('count');

        }
        else {
            if( $wc_coupons_id ) foreach( $wc_coupons_id as $wc_coupon_id ) {
                $WC_Coupon = new \WC_Coupon( $wc_coupon_id );
                if( !$WC_Coupon ) {
                    continue;
                }


                $timestamp = strtotime( $WC_Coupon->get_date_modified()->date_i18n( 'Y-m-d H:i:s' ) );
                if( $lastUpdate && $timestamp <= $lastUpdate ) {
                    //	$vouchers[] = "skipping $wc_coupon_id because " . date('Y-m-d H:i:s', $timestamp) . " <= " . date('Y-m-d H:i:s', $lastUpdate);
                    continue;
                }

                $type_voucher 	= $WC_Coupon->get_discount_type();
                $shopimind_type_voucher = shopimind_get_type_voucher( $type_voucher );

                $date_from 	= $WC_Coupon->get_date_created() ? $WC_Coupon->get_date_created()->date_i18n( 'Y-m-d H:i:s' ) : '0000-00-00 00:00:00';
                $date_to 	= $WC_Coupon->get_date_expires() ? $WC_Coupon->get_date_expires()->date_i18n( 'Y-m-d H:i:s' ) : '0000-00-00 00:00:00';
                $voucher 	= array(
                    'id_shop'					=> intval( $this->getShopID() ),
                    'name'						=> $WC_Coupon->get_description(),
                    'type_voucher'				=> $shopimind_type_voucher,
                    //'amount'					=> $shopimind_type_voucher == 2 ? $WC_Coupon->get_amount() : false,
                    'amount'					=> $WC_Coupon->get_amount(),
                    'reduction_tax'				=> $WC_Coupon->apply_before_tax() ? 0 : 1,
                    'used'						=> $WC_Coupon->get_usage_count() ? 1 : 0,
                    'id_voucher'				=> intval( $WC_Coupon->get_id() ),
                    'code'						=> $WC_Coupon->get_code(),
                    'date_from'					=> $date_from,
                    'date_to'					=> $date_to,
                    'active'					=> 1,
                    'id_customer_shop'			=> 0,
                    'date_creation'				=> $WC_Coupon->get_date_created()->date_i18n( 'Y-m-d H:i:s' ),
                    'date_update'				=> $WC_Coupon->get_date_modified()->date_i18n( 'Y-m-d H:i:s' ),
                    'minimum_amount'			=> floatval( $WC_Coupon->get_minimum_amount() ),
                    'reduction_currency'		=> \get_woocommerce_currency(), //\html_entity_decode( \get_woocommerce_currency_symbol() ),
                    'minimum_amount_currency'	=> \get_woocommerce_currency(), //\html_entity_decode( \get_woocommerce_currency_symbol() ),
                    'lang'						=> substr( \get_locale() , 0, 2 ),
                    //'lastUpdate'				=> date( 'Y-m-d H:i:s' ),
                );

                $restrictions = $WC_Coupon->get_email_restrictions();
                if( count( $restrictions ) > 1 ) {

                }
                elseif( $restrictions ) {
                    $WP_User = \get_user_by( 'email', $restrictions[0] );
                    $voucher['id_customer_shop'] = $WP_User->ID;
                }
                else {
                    $voucher['id_customer_shop']  = 0;
                }

                if( $timestamp > $lastUpdate ) {
                    //$lastUpdate = $timestamp;
                }
                $vouchers[] = $voucher;
            }
        }

        //$vouchers[] = " args last UPdate = " . $args['lastUpdate'] . " / " . get_gmt_from_date( $args['lastUpdate'] ) . " = strtotime = " . strtotime( $args['lastUpdate'] ) . " / " . strtotime( get_gmt_from_date( $args['lastUpdate'] )) ;

        if( isset ($args['start'] ) && isset( $args['limit'] ) ) {
            $vouchers = array_slice( $vouchers, $args['start'], $args['limit'] );
        }

        $lastUpdate = $lastUpdate ? date( 'Y-m-d H:i:s', $lastUpdate ) : date( 'Y-m-d H:i:s');
        $return = compact( 'lastUpdate', 'vouchers' );
        return \apply_filters( __METHOD__,  $return, $args );
    }

    /** CUSTOMERS **/

    /**
     * Renvoie la liste des ID des groupes utilisateurs
     * @param	null
     * @return	array ( customersGroups => array( idgroup => name ) )
     */
    public function getCustomerGroups( $args ) {
        // Get Roles
        if ( !function_exists( 'get_editable_roles' ) ) {
            require_once ABSPATH . 'wp-admin/includes/user.php';
        }


        $blog_ids = $this->getBlogIds( $args);
        $customersGroups = array();

        $blog_id = array_pop( $blog_ids );
        if( function_exists( 'switch_to_blog' ) && \get_current_blog_id() != $blog_id ) {
            \switch_to_blog( $blog_id );
        }


        $roles = \get_editable_roles();
        if( !$roles ) {
            return new APIError( 'execution', __( "Aucun groupe client défini", 'shopimind' ) );
        }


        foreach( $roles as $role => $role_data ) {
            $customersGroups[$role] = $role_data['name'];
        }

        if( function_exists( 'restore_current_blog' ) ) {
            \restore_current_blog();
        }

        $return = compact( 'customersGroups' );
        return \apply_filters( __METHOD__,  $return );
    }

    /**
     * Renvoie une liste détaillée des groupes d'utilisateurs
     * Permet de récupérer la liste des groupes de contacts présents dans la base
     * Cette méthode est utilisée pour la fonctionnalité de campagne SMS et Email
     * @param	timestamp 	$start
     * @param	integer 	$limit
     * @param	timestamp 	$lastUpdate
     * @param	integer 	$idGroup
     * @param	bool 		$justCount
     * @return	array 		( lastUpdate => timestamp, customersGroups => array( id_group, id_shop, name, lang, date_creation, date_update ) )
     *
     */
    public function syncCustomersGroups( $args ) {
        // Get Roles
        if ( !function_exists( 'get_editable_roles' ) ) {
            require_once ABSPATH . 'wp-admin/includes/user.php';
        }
        $roles = \get_editable_roles();

        if( isset( $args['idGroup'] ) && !empty( $args['idGroup'] )) {
            $idGroup = $args['idGroup'];

            foreach( $roles as $role => $role_data ) {
                if( $role != $idGroup ) {
                    unset( $roles[$role] );
                }
            }
        }


        $customersGroups = array();
        if( isset($args['justCount']) && $args['justCount'] ) {
            $customersGroups = array( 'count' => is_array($roles) ? count($roles) : 0 );
        }
        else {
            foreach( $roles as $role => $role_data ) {
                $customerGroup = array(
                    'id_group'	=> $role,
                    'id_shop'	=> intval( $this->getShopID() ),
                    'name'		=> $role_data['name'],
                    'lang'		=> substr( \get_locale(), 0, 2 ),
                    'date_creation'	=> '',
                    'date_update'	=> '',
                );
                $customersGroups[] = $customerGroup;
            }

            if( isset ($args['start'] ) && isset( $args['limit'] ) ) {
                $customersGroups = array_slice( $customersGroups, $args['start'], $args['limit'] );
            }
        }

        $lastUpdate = '';
        //$lastUpdate = $lastUpdate ? date( 'Y-m-d H:i:s', $lastUpdate ) : date( 'Y-m-d H:i:s');
        $return = compact( 'lastUpdate', 'customersGroups' );
        return \apply_filters( __METHOD__,  $return );
    }
    /**
     * Renvoie la liste des ids clients modifiés depuis $lastUpdate
     * @param	timestamp 	$start
     * @param	integer 	$limit
     * @param	timestamp 	$lastUpdate
     * @param	integer 	$idCustomer
     * @param	bool 		$justCount
     * @return	array 		( lastUpdate => timestamp, customers => array( idcustomer ) )
     *
     */
    public function syncCustomers( $args ) {
        $user_args = array(
            'number'	=> -1,
            'role__not_in' => array( 'administrator')
        );

        if (version_compare(WC()->version, '4.2.0', '>=')){
            $user_args['orderby'] = array(
                'last_update' => 'ASC',
                'ID' => 'ASC'
            );
        }else {
            $user_args['orderby'] = 'ID';
            $user_args['order'] = 'ASC';
        }

        $customers = array();
        $justCount = isset($args['justCount']) && $args['justCount'];

        if (isset($args['limit']) && intval($args['limit'])) {
            $user_args['number'] = $args['limit'];
        }
        if (isset($args['start']) && $args['start'] ) {
            $user_args['offset'] = $args['start'];
        }
        if (isset($args['idCustomer']) && intval($args['idCustomer'])) {
            $user_args['include'] = array(intval($args['idCustomer']));
        }

        if (isset($args['lastUpdate']) && $args['lastUpdate'] && $args['lastUpdate'] != "") {
            $updated_since = strtotime(get_gmt_from_date( $args['lastUpdate']));
            $meta_query_args = array(
                array(
                    'relation' => 'OR',
                    array(
                        'key'     => 'user_registered',
                        'compare' => '>=',
                        'value' => $updated_since,
                    ),
                    array(
                        'key'     => 'last_update',
                        'compare' => '>=',
                        'value'   => $updated_since,
                    )
                )
            );
            $user_args['meta_query'] = $meta_query_args;
        }

        if ($justCount) {
            $user_args['number'] = 1;
            $user_args['paged'] = 1;
        }

        $blog_ids = $this->getBlogIds( $args );

        foreach($blog_ids as $blog_id ) {
            if( function_exists( 'switch_to_blog' ) && \get_current_blog_id() != $blog_id ) {
                \switch_to_blog( $blog_id );
            }

            $wp_users = new \WP_User_Query($user_args);

            if ($justCount) {
                $customers = array('count' => $wp_users->get_total() !== null ? $wp_users->get_total() : 0);
            } else if (is_array($wp_users->get_results())) {
                foreach ($wp_users->get_results() as $WP_User) {
                    $customers[$WP_User->ID] = $this->formatUser(array(
                        'idUser' 		=> $WP_User->ID,
                        'shopIdShop'	=> $blog_id,
                    ));
                }
            }
        }

        if( function_exists( 'restore_current_blog' ) ) {
            \restore_current_blog();
        }

        $lastUpdate = date('Y-m-d H:i:s');
        $return = compact( 'lastUpdate', 'customers' );
        return \apply_filters( __METHOD__,  $return );
    }

    /**
     * Génére un compte client
     * Création d'un nouveau customer sur la boutique
     * @param 	array 	customer ( email, lastName, firstName, password, birthday, newsletter )
     * @return 	array 	( success => bool )
     */
    public function createCustomer( $args ) {
        $shopIdShop = $args['shopIdShop'];

        // TODO : switch to proper blog id

        $customer = $args['customer'];

        $email = $customer['email'];
        $username = $customer['firstName'] . ' ' . $customer['lastName'];
        $username = \sanitize_user( $username );
        $raw_username = $username;
        $i = 2;
        while( \username_exists( $username ) ) {
            $username = $raw_username . $i;
            $i++;
        }
        $password = isset( $customer['password'] ) ? $customer['password'] : '';

        // Bypass validate_privacy_policy_checkbox
        remove_action( 'woocommerce_register_post', 'validate_privacy_policy_checkbox', 10 );

        if ( ! isset( $_POST['privacy_policy'] ) ) {
            $_POST['privacy_policy'] = 'on';
        }

        $user_id = \wc_create_new_customer( $email, $username, $password );
        if( \is_wp_error( $user_id ) ) {
            return new APIError( 'user', $user_id->get_error_message() );
        }

        \update_user_meta( $user_id, "first_name", $customer['firstName'] );
        \update_user_meta( $user_id, "last_name", $customer['lastName'] );

        \update_user_meta( $user_id, "billing_first_name", $customer['firstName'] );
        \update_user_meta( $user_id, "shipping_first_name", $customer['firstName'] );

        \update_user_meta( $user_id, "billing_last_name", $customer['lastName'] );
        \update_user_meta( $user_id, "shipping_last_name", $customer['lastName'] );

        $birthday_field = \ShopiMindConfiguration::getUserBirthdayField();
        if( isset( $customer['birthday'] ) && $birthday_field != 'none' ) {
            \update_user_meta( $user_id, $birthday_field, $customer['birthday'] );
        }
        $newsletter_field = \ShopiMindConfiguration::getUserNewsletterField();
        if( isset( $customer['newsletter'] ) && $newsletter_field != 'none' ) {
            \update_user_meta( $user_id, $newsletter_field, $customer['newsletter'] );
        }

        $return = array(
            'user_id'	=> intval( $user_id ),
            'username'	=> $username,
        );
        return \apply_filters( __METHOD__,  $return, $args );
    }

    /**
     * Renvoie les données détaillées d'un compte utilisateur
     * @param 	array $args
     * @return 	array|APIError	( user => array( id_customer, shop_id_shop, locale, first_name, last_name, birthday, email, optin, newsletter, customer_since, customer_update, gender, groups, active, addresses, custom ) )
     */
    public function formatUser( $args ) {
        $idUser = isset( $args['idUser'] ) ? $args['idUser'] : false;
        $WC_Customer = isset( $args['WC_Customer'] ) ? $args['WC_Customer'] : false;

        if (!$WC_Customer && $idUser) {
            $WC_Customer = new \WC_Customer($idUser);
        }

        if (!$WC_Customer || is_null($WC_Customer->get_date_created()) || is_null($WC_Customer->get_date_modified())) {
            return new APIError( 'execution', sprintf( __( "Utilisateur inconnu : '%d'", 'shopimind' ), $idUser ) );
        }

        $user_blog_ids = wp_list_pluck(get_blogs_of_user( $WC_Customer->get_id() ), 'userblog_id');

        if( isset( $args['shopIdShop'] ) && $args['shopIdShop'] && !in_array($args['shopIdShop'], $user_blog_ids)) {
            return new APIError( 'execution', sprintf( __( "Utilisateur %d inconnu sur ce site %d", 'shopimind' ), $idUser, $args['shopIdShop'] ) );
        }
        $userdata = get_userdata($WC_Customer->get_id());

        $customer_since = !is_null($WC_Customer->get_date_created()) ? get_date_from_gmt($WC_Customer->get_date_created()->date_i18n('Y-m-d H:i:s')) : "0000-00-00 00:00:00";


        // ---------- Locale (langue + pays) du client via Polylang + cache ----------
        $customer_locale = ''; // init pour éviter tout notice

        if ( function_exists('pll_get_post_language') ) {
            // Lire le cache
            $customer_locale = \get_user_meta( $WC_Customer->get_id(), 'spm_locale', true );

            if ( empty($customer_locale) ) {
                // Dernière commande du client -> locale de la commande
                $order_ids = \wc_get_orders( array(
                    'customer' => $WC_Customer->get_id(),
                    'limit'    => 1,
                    'orderby'  => 'date',
                    'order'    => 'DESC',
                    'return'   => 'ids',
                ) );

                if ( ! empty($order_ids) ) {
                    $locale_from_order = pll_get_post_language( $order_ids[0], 'locale' ); // ex: fr_FR
                    if ( ! empty($locale_from_order) ) {
                        $customer_locale = $locale_from_order;
                    }
                }
            }
        }

        if ( empty($customer_locale) ) {
            // Locale courante si Polylang, sinon WP
            if ( function_exists('pll_current_language') ) {
                $customer_locale = pll_current_language('locale'); // ex: en_GB
            }
            if ( empty($customer_locale) ) {
                $customer_locale = \get_locale(); // fallback WP, ex: fr_FR
            }
        }

        // Cache le locale déduit pour la prochaine fois (si Polylang actif)
        if ( function_exists('pll_current_language') && empty( \get_user_meta( $WC_Customer->get_id(), 'spm_locale', true ) ) && ! empty($customer_locale) ) {
            \update_user_meta( $WC_Customer->get_id(), 'spm_locale', $customer_locale );
        }
        // --------------------------------------------------------------------------



        $user = array(
            'id_customer'	    => intval($WC_Customer->get_id()),
            'shop_id_shop'	    => (isset($args['shopIdShop']) && $args['shopIdShop']) ?
                $args['shopIdShop'] :
                array_pop( $user_blog_ids ),
            'locale'		    => $customer_locale,
            'first_name'	    => isset($userdata->user_firstname) ?
                $userdata->user_firstname :
                get_user_meta($WC_Customer->get_id(), 'billing_first_name', true),

            'last_name'		    => isset($userdata->user_lastname) ?
                $userdata->user_lastname :
                get_user_meta($WC_Customer->get_id(), 'billing_last_name', true),

            'birthday'		    => '0000-00-00',
            'email'			    => $WC_Customer->get_email(),
            'optin'			    => 1,
            'newsletter'	    => 1,
            'customer_since'	=> $customer_since,
            'customer_update'	=> ($WC_Customer->get_date_modified()) ?
                $WC_Customer->get_date_modified()->date_i18n('Y-m-d H:i:s') :
                get_date_from_gmt($WC_Customer->get_date_created()->date_i18n('Y-m-d H:i:s')),

            'gender'		    => 0,
            'groups'		    => array($WC_Customer->get_role()),
            'active'		    => 1,
            'addresses'		    => array(),
            'custom'		    => array(),
        );

        if (version_compare(WC()->version, '3.2.0', '>=')){
            $billing_address = $WC_Customer->get_billing();
            $shipping_address = $WC_Customer->get_shipping();
        } else {
            $billing_address = array(
                'first_name'	=> $WC_Customer->get_billing_first_name(),
                'last_name'	=> $WC_Customer->get_billing_last_name(),
                'phone'	=> $WC_Customer->get_billing_phone(),
                'company'	=> $WC_Customer->get_billing_company(),
                'email' => $WC_Customer->get_billing_email(),
                'country'	=> $WC_Customer->get_billing_country(),
                'city'		=> $WC_Customer->get_billing_city(),
                'state'		=> $WC_Customer->get_billing_state(),
                'address_1'	=> $WC_Customer->get_billing_address(),
                'address_2'	=> $WC_Customer->get_billing_address_2(),
                'postcode'	=> $WC_Customer->get_billing_postcode(),
            );
            $shipping_address = array(
                'first_name'	=> $WC_Customer->get_shipping_first_name(),
                'last_name'	=> $WC_Customer->get_shipping_last_name(),
                'company'	=> $WC_Customer->get_shipping_company(),
                'country'	=> $WC_Customer->get_shipping_country(),
                'city'		=> $WC_Customer->get_shipping_city(),
                'state'		=> $WC_Customer->get_shipping_state(),
                'address_1'	=> $WC_Customer->get_shipping_address(),
                'address_2'	=> $WC_Customer->get_shipping_address_2(),
                'postcode'	=> $WC_Customer->get_shipping_postcode(),
            );
        }

        if (implode('', $billing_address ) !== '') {
            $formattedBillingAddress = getAddress( $billing_address );
            $user['addresses'][] = $formattedBillingAddress;
        }
        if (implode( '', $shipping_address ) !== '') {
            $formattedShippingAddress = getAddress( $shipping_address );
            if (!isset($formattedBillingAddress) || $formattedShippingAddress != $formattedBillingAddress) {
                $user['addresses'][] = $formattedShippingAddress;
            }
        }

        $birthday_field = \ShopiMindConfiguration::getUserBirthdayField();
        if( $birthday_field && $birthday_field !== 'none' ) {
            if( function_exists('get_field')) {
                $birthday = get_field($birthday_field, 'user_' . $WC_Customer->get_id() );
            }
            else {
                $birthday = $WC_Customer->get_meta( $birthday_field );
            }

            if($birthday) {
                $birthday_date = \DateTime::createFromFormat('Y-m-d', $birthday);
                if( !$birthday_date ) {
                    $birthday_date = \DateTime::createFromFormat('Ymd', $birthday);
                }
                if( !$birthday_date ) {
                    $birthday_date = \DateTime::createFromFormat('d/m/Y', $birthday);
                }
                if( !$birthday_date ) {
                    $birthday = false;
                }
                else {
                    $birthday = $birthday_date->format('Y-m-d H:i:s');
                }
            }
            $user['birthday'] = $birthday ? $birthday : '0000-00-00 00:00:00';
        }

        $gender_field = \ShopiMindConfiguration::getUserGenderField();
        if( $gender_field && $gender_field !== 'none' ) {
            if( function_exists('get_field')) {
                $gender = get_field($gender_field, 'user_' . $WC_Customer->get_id() );
            }
            else {
                $gender = $WC_Customer->get_meta( $gender_field );
            }

            $user['gender'] = $gender ? $gender : 0;
        }

        $newsletter_field = \ShopiMindConfiguration::getUserNewsletterField();
        if( $newsletter_field && $newsletter_field !== 'none' ) {
            $user['newsletter'] = intval( $WC_Customer->get_meta( $newsletter_field ) );
        }

        return \apply_filters( __METHOD__, $user);
    }

    /** END CUSTOMERS **/

    /** CARRIERS **/
    /**
     * Renvoie la liste des instances d'expédition ?
     * @param	null
     * @return	array( carriers => array( instance_id => name ) )
     * @internal how to manage zones ?
     *
     */
    public function getCarriers() {

        return array(
            'success' => true,
            'carriers'	=> array()
        );

        $instances = \shopimind_get_possible_shippings();
        if( !$instances ) {
            return new APIError( 'execution', __( "Aucune méthode d'expédition configurée", 'shopimind' ) );
        }

        $carriers = array();

        if( $instances ) foreach( $instances as $instance_id	=> $instance ) {
            if( !empty( $instance['zone_name'] ) ) {
                $instance_title = sprintf( __( '%s ( %s )', 'shopimind' ), $instance['title'], $instance['zone_name'] );
            }
            else {
                $instance_title = sprintf( __( '%s ', 'shopimind' ), $instance['title'] );
            }
            $carriers[$instance_id] = $instance_title;
        }

        $success = true;
        $return = compact( 'carriers', 'success' );
        return \apply_filters( __METHOD__,  $return );
    }
    /**
     * Renvoie une liste détaillée des instances d'expédition
     * @internal il faudra étudier l'approche transporteurs versus approche instance d'expédition de WooCommerce
     *
     * @param 	integer $idCarrier
     * @param 	boolean $justCount
     * @param   integer $start // ajouté 01/2020
     * @param 	integer $limit // ajoute 01/2020
     * @return 	array 	( lastUpdate => timestmap, carriers => array( id_carrier, id_shop, name, active ) )
     */
    public function syncCarriers( $args ) {
        $instances = \shopimind_get_possible_shippings();

        $carriers = array();

        if( $instances ) foreach( $instances as $instance_id	=> $instance ) {
            if( !empty( $instance['zone_name'] ) ) {
                $instance_title = sprintf( __( '%s ( %s )', 'shopimind' ), $instance['title'], $instance['zone_name'] );
            }
            else {
                $instance_title = sprintf( __( '%s ', 'shopimind' ), $instance['title'] );
            }
            $carriers[] = array(
                'id_carrier'	=> intval( $instance_id ),
                'id_shop'		=> intval( $this->getShopID() ),
                'name'			=> $instance_title,
                'active'		=> 1,
                'last_update'	=> date( 'Y-m-d H:i:s'),
            );
        }

        if( isset ($args['start'] ) && isset( $args['limit'] ) ) {
            $carriers = array_slice( $carriers, $args['start'], $args['limit'] );
        }

        if( isset($args['justCount']) && $args['justCount'] ) {
            $carriers = array( 'count' => is_array($carriers) ? count($carriers) : 0 );
        }

        //$lastUpdate = '';
        $lastUpdate = isset($args['lastUpdate']) && $args['lastUpdate'] ? date( 'Y-m-d H:i:s', $args['lastUpdate'] ) : date( 'Y-m-d H:i:s');
        $return = compact( 'lastUpdate', 'carriers' );
        return \apply_filters( __METHOD__,  $return );
    }

    /** END CARRIERS **/

    /** FABRICANTS **/
    /**
     * Renvoie une liste détaillée de fabricants
     * @param	timestamp 	$start
     * @param	integer 	$limit
     * @param	timestamp 	$lastUpdate
     * @param	integer 	$idManufacturer
     * @param	bool 		$justCount
     * @return 	array 		( lastUpdate => timestamp, manufacturers=> array( id_manufacturer, id_shop, name, active, date_creation, date_update ) )
     */
    public function syncManufacturers( $args ) {
        $manufacturers_list = false;

        $manufacturer_field_type = \ShopiMindConfiguration::getProductManufacturerFieldType();
        $meta_key = $taxonomy = false;
        if( $manufacturer_field_type == 'meta' ) {
            $meta_key = \ShopiMindConfiguration::getProductManufacturerField();
        }
        elseif( $manufacturer_field_type == 'taxonomy' ) {
            $taxonomy = \ShopiMindConfiguration::getProductManufacturerTaxonomy();
        }
        if( $meta_key ) {
            $manufacturers_list = getDistinctMetaValuesFor( $meta_key, false );
        }
        if( $taxonomy ) {
            $manufacturers_list = getDistinctTermsFor( $taxonomy, false );
        }

        $manufacturers = array();
        if( isset($args['justCount']) && $args['justCount'] ) {
            $manufacturers = array( 'count' => is_array($manufacturers_list) ? count($manufacturers_list) : 0 );
        }
        if( $manufacturers_list ) foreach( $manufacturers_list as $wc_manufacturer ) {
            $manufacturer = array(
                'id_manufacturer' 	=> ( isset( $wc_manufacturer['ID'] )) ? $wc_manufacturer['ID'] : $wc_manufacturer['name'],
                'id_shop'			=> intval( $this->getShopID() ),
                'name'				=> $wc_manufacturer['name'],
                'active'			=> 1,
                'date_creation'		=> '',
                'date_update'		=> '',
            );
            $manufacturers[] = $manufacturer;
        }

        if( isset ($args['start'] ) && isset( $args['limit'] ) ) {
            $manufacturers = array_slice( $manufacturers, $args['start'], $args['limit'] );
        }


        $lastUpdate = '';
        $return = compact( 'lastUpdate', 'manufacturers' );
        return \apply_filters( __METHOD__,  $return, $args );
    }

    /** END FABRICANTS **/

    protected function getBlogIds( $args ) {
        if(isset($args['shopIdShop']) && $args['shopIdShop']) {
            $blog_ids = array($args['shopIdShop']);
        }
        elseif( \ShopiMindConfiguration::isMultiShops() ) {
            $blog_ids = \get_sites(array('fields' => 'ids') );
        }
        else {
            $blog_ids = array( get_current_blog_id() );
        }
        return $blog_ids;
    }

    /** PRODUITS **/

    /**
     * Renvoie une liste détaillée de catégories de produits
     * @param	timestamp 	$start
     * @param	integer 	$limit
     * @param	timestamp 	$lastUpdate
     * @param	integer 	$idCategory
     * @param	bool 		$justCount
     * @return 	array 		( lastUpdate => timestamp, productsCategories => array( shop_id_shop, id_category, id_parent_category, lang, name, description, link, date_creation, date_update, active ) )
     */
    public function syncProductsCategories( $args ) {
        $cat_args = array(
            'hide_empty' => false,
        );
        if ( isset( $args['start'] ) && isset( $args['limit'] ) ) {
            $cat_args['offset'] = $args['start'];
            $cat_args['number'] = $args['limit'];
        }
        if ( isset( $args['idCategory'] ) && ! empty( $args['idCategory'] ) && intval( $args['idCategory'] ) ) {
            $cat_args['term_taxonomy_id'] = intval( $args['idCategory'] );
        }

        $blog_ids = $this->getBlogIds( $args );
        $productsCategories = array();

        foreach ( $blog_ids as $blog_id ) {
            if (function_exists('switch_to_blog') && \get_current_blog_id() != $blog_id) {
                \switch_to_blog($blog_id);
            }

            global $wpdb;

            // 1. Si justCount → compter uniquement les catégories en langue par défaut
            if (!empty($args['justCount'])) {
                global $wpdb;

                $default_lang = function_exists('pll_default_language') ? pll_default_language() : substr(get_locale(), 0, 2);

                // Récupérer tous les term_taxonomy_id pour la langue par défaut
                $all_term_ids = get_terms('product_cat', ['hide_empty' => false, 'fields' => 'ids']);
                $term_tax_ids = [];

                foreach ($all_term_ids as $term_id) {
                    if (function_exists('pll_get_term_language') && pll_get_term_language($term_id) !== $default_lang) {
                        continue;
                    }
                    $term_tax_id = $wpdb->get_var($wpdb->prepare("
			SELECT term_taxonomy_id
			FROM {$wpdb->term_taxonomy}
			WHERE term_id = %d AND taxonomy = 'product_cat'
		", $term_id));

                    if (!$term_tax_id) continue;

                    $term_tax_ids[] = $term_tax_id;
                }

                // Appliquer un filtre lastUpdate si fourni
                if (!empty($args['lastUpdate'])) {
                    $gmt_date = esc_sql(get_gmt_from_date($args['lastUpdate']));

                    $term_tax_ids = array_filter($term_tax_ids, function ($ttid) use ($wpdb, $gmt_date) {
                        $result = $wpdb->get_var("
				SELECT 1
				FROM {$wpdb->term_relationships} tr
				JOIN {$wpdb->posts} p ON p.ID = tr.object_id
				WHERE tr.term_taxonomy_id = $ttid
				AND p.post_type = 'product'
				AND p.post_status IN ('publish', 'draft', 'private')
				AND p.post_modified_gmt >= '$gmt_date'
				LIMIT 1
			");
                        return !empty($result);
                    });
                }

                $productsCategories = ['count' => count($term_tax_ids)];
            } else {
                // 2. Si lastUpdate défini → filtrer les catégories concernées

                if (!empty($args['lastUpdate'])) {
                    $term_ids_filtered_by_update = [];
                    $gmt_date = esc_sql(get_gmt_from_date($args['lastUpdate']));
                    $term_ids_filtered_by_update = $wpdb->get_col("
                SELECT DISTINCT tr.term_taxonomy_id
                FROM {$wpdb->term_relationships} tr
                JOIN {$wpdb->posts} p ON p.ID = tr.object_id
                JOIN {$wpdb->term_taxonomy} tt ON tr.term_taxonomy_id = tt.term_taxonomy_id
                WHERE tt.taxonomy = 'product_cat'
                AND p.post_type = 'product'
                AND p.post_status IN ('publish', 'draft', 'private')
                AND p.post_modified_gmt >= '$gmt_date'
            ");
                }

                // 3. Récupération des catégories WordPress
                $wc_categories = get_terms('product_cat', $cat_args);

                foreach ($wc_categories as $WP_Term) {
                    $term_tax_id = $wpdb->get_var( $wpdb->prepare("
    SELECT term_taxonomy_id
    FROM {$wpdb->term_taxonomy}
    WHERE term_id = %d AND taxonomy = 'product_cat'
", $WP_Term->term_id ) );

                    if (isset($term_ids_filtered_by_update) && !in_array($term_tax_id, $term_ids_filtered_by_update)) {
                        continue;
                    }

                    // Polylang : on ne traite la catégorie qu’en langue par défaut
                    if (function_exists('pll_get_term_language') && function_exists('pll_default_language')) {
                        $default_lang = pll_default_language();
                        if (pll_get_term_language($WP_Term->term_id) !== $default_lang) {
                            continue;
                        }

                        $translations = pll_get_term_translations($WP_Term->term_id);
                    } else {
                        $translations = [substr(get_locale(), 0, 2) => $WP_Term->term_id];
                    }

                    foreach ($translations as $lang => $translated_term_id) {
                        $translated_term = get_term($translated_term_id, 'product_cat');
                        if (!$translated_term || is_wp_error($translated_term)) {
                            continue;
                        }

                        if (function_exists('pll_set_language')) {
                            pll_set_language($lang);
                        }

                        // Récupérer dates approximées via les posts liés
                        $term_id = $translated_term->term_id;

                        $default_lang = function_exists('pll_default_language') ? pll_default_language() : substr(get_locale(), 0, 2);
                        $translations = function_exists('pll_get_term_translations') ? pll_get_term_translations($WP_Term->term_id) : [$default_lang => $WP_Term->term_id];

                        $default_term_id = isset($translations[$default_lang]) ? $translations[$default_lang] : $WP_Term->term_id;
                        $default_term_taxonomy_id = $wpdb->get_var($wpdb->prepare("
                        SELECT term_taxonomy_id FROM {$wpdb->term_taxonomy}
                        WHERE term_id = %d AND taxonomy = 'product_cat'
                    ", $default_term_id));

                        // Dates calculées sur la catégorie par défaut (valables pour toutes les traductions)
                        $date_creation = $wpdb->get_var($wpdb->prepare("
                        SELECT MIN(p.post_date)
                        FROM {$wpdb->term_relationships} tr
                        JOIN {$wpdb->posts} p ON p.ID = tr.object_id
                        WHERE tr.term_taxonomy_id = %d
                    ", $default_term_taxonomy_id));

                        $date_update = $wpdb->get_var($wpdb->prepare("
                        SELECT MAX(p.post_modified)
                        FROM {$wpdb->term_relationships} tr
                        JOIN {$wpdb->posts} p ON p.ID = tr.object_id
                        WHERE tr.term_taxonomy_id = %d
                    ", $default_term_taxonomy_id));

                        if (!$date_creation) $date_creation = \ShopiMindConfiguration::getPluginInstallDate();
                        if (!$date_update) $date_update = $date_creation;

                        $productCategory = array(
                            'shop_id_shop' => $this->getShopID(),
                            'id_category' => intval($WP_Term->term_taxonomy_id),
                            'id_parent_category' => intval($translated_term->parent),
                            'lang' => $lang,
                            'name' => $translated_term->name,
                            'description' => $translated_term->description,
                            'link' => get_term_link($translated_term, 'product_cat'),
                            'date_creation' => date('Y-m-d H:i:s', strtotime($date_creation)),
                            'date_update' => date('Y-m-d H:i:s', strtotime($date_update)),
                            'active' => 1,
                        );

                        $productsCategories[] = $productCategory;
                    }
                }
            }
        }
        if ( function_exists( 'restore_current_blog' ) ) {
            restore_current_blog();
        }

        $lastUpdate = date( 'Y-m-d H:i:s' );
        return apply_filters( __METHOD__, compact('lastUpdate', 'productsCategories') );
    }



    /**
     * Renvoie une liste détaillée de produits
     * @param	timestamp 	$start
     * @param	integer 	$limit
     * @param	timestamp 	$lastUpdate
     * @param	integer 	$idProduct
     * @param	bool 		$justCount
     * @return 	array 		( lastUpdate => timestamp, products => array( shop_id_shop, id_product, reference, lang, name, combination_name, description_short, description, product_link, image_link, images, id_combination, id_categories, id_manufacturer, currency, price, price_discount, quantity_remaining, date_creation, date_update, active, custom ) )
     */

    public function syncProducts( $args ) {
        $product_args = array(
            'post_type'      => 'product',
            'orderby' => 'modified',
            'order' => 'DESC',
            'parent' => 0
        );

        $products = array();
        $justCount = isset($args['justCount']) && $args['justCount'];

        if (isset($args['limit'])) {
            $product_args['limit'] = $args['limit'];
        }
        if (isset($args['start']) && $args['start']) {
            $product_args['offset'] = $args['start'];
        }
        if (isset($args['lastUpdate'] ) && $args['lastUpdate'] && $args['lastUpdate'] != "") {
            $product_args['date_modified'] = ">=" . strtotime(get_gmt_from_date($args['lastUpdate']));
        }
        if (isset($args['idProduct']) && intval($args['idProduct'])) {
            $product_args['include'] = array(intval($args['idProduct']));
        }

        if ($justCount) {
            $product_args['limit'] = 1;
            $product_args['paginate'] = true;
        }

        $blog_ids = $this->getBlogIds( $args );

        foreach($blog_ids as $blog_id ) {
            if( function_exists( 'switch_to_blog' ) && \get_current_blog_id() != $blog_id ) {
                \switch_to_blog( $blog_id );
            }

            $wc_products = \wc_get_products($product_args);

            if ($justCount) {
                $products = array('count' => isset($wc_products->total) ? $wc_products->total : 0);
            } else if ($wc_products && is_array($wc_products)){
                foreach ($wc_products as $WC_Product) {
                    if($WC_Product->is_type( 'variation' ))
                        continue;
                    /* Formater le produit */
                    foreach ( $this->formatProduct( $WC_Product ) as $productData ) {
                        $products[] = $productData;
                    }
                }
            }
        }

        if( function_exists( 'restore_current_blog' ) ) {
            \restore_current_blog();
        }

        $lastUpdate = date('Y-m-d H:i:s');
        $return = compact( 'lastUpdate', 'products' ); // 'product_args'
        return \apply_filters( __METHOD__,  $return );
    }

    /** END PRODUITS **/

    /** COMMANDES **/

    /**
     * Renvoie la liste des statuts de commande
     * @param	null
     * @return	array 	( status => array( idstatus => name ) )
     */
    public function orderStatus() {
        $order_statuses = \wc_get_order_statuses();

        $status = array();
        if( $order_statuses ) foreach( $order_statuses as $idstatus => $name ) {
            $status[$idstatus] = $name;
        }

        $return = compact( 'status' );
        return \apply_filters( __METHOD__,  $return );
    }

    /**
     * Renvoie une liste détaillée des statuts des commandes correspondantes aux paramètres d'entrée
     * Permet de récupérer la liste des statuts de commandes présents dans la base
     * Cette méthode est utilisée pour la fonctionnalité de campagne SMS et Email
     * @param  timestamp $start
     * @param  integer   $limit
     * @param  timestamp $lastUpdate  // (sans effet ici : pas de notion de MAJ sur les labels)
     * @param  string    $idStatus    filtre les commandes portant cet identifiant de statut
     * @param  bool      $justCount
     * @return array     ( lastUpdate => timestamp, ordersStatus => array( id_status, name, lang, deleted ) )
     */
    public function syncOrdersStatus( $args ) {
        $wc_statuses = \wc_get_order_statuses();

        // --- justCount : calcule le nombre d'éléments qui seraient renvoyés (statuts filtrés × langues) ---
        if ( ! empty( $args['justCount'] ) ) {
            // Langues (slugs) — même logique que dans la liste, mais sans switch / reload
            if ( function_exists('pll_languages_list') ) {
                $langs = \pll_languages_list( [ 'fields' => 'slug' ] );
                if ( empty( $langs ) ) {
                    $langs = [ substr( \get_locale(), 0, 2 ) ];
                }
            } else {
                $langs = [ substr( \get_locale(), 0, 2 ) ];
            }

            // Statuts filtrés (par clé)
            $filtered_status_keys = array_keys( $wc_statuses );
            if ( ! empty( $args['idStatus'] ) ) {
                if ( isset( $wc_statuses[ $args['idStatus'] ] ) ) {
                    $filtered_status_keys = [ $args['idStatus'] ];
                } else {
                    $filtered_status_keys = [];
                }
            }

            $count = count( $filtered_status_keys ) * count( $langs );

            $lastUpdate   = date('Y-m-d H:i:s');
            $ordersStatus = [ 'count' => $count ];
            return \apply_filters( __METHOD__, compact('lastUpdate','ordersStatus'), $args );
        }

        // Langues cibles (slugs) et locales correspondantes
        if ( function_exists('pll_languages_list') ) {
            $langs   = \pll_languages_list( [ 'fields' => 'slug' ] );   // ex: ['fr','en']
            $locales = \pll_languages_list( [ 'fields' => 'locale' ] ); // ex: ['fr_FR','en_GB']
            if ( empty($langs) )  { $langs  = [ substr(\get_locale(), 0, 2) ]; }
            if ( empty($locales) ){ $locales= [ \get_locale() ]; }
        } else {
            $langs   = [ substr(\get_locale(), 0, 2) ];
            $locales = [ \get_locale() ];
        }

        $ordersStatus = [];
        $deleted_map  = ['wc-cancelled', 'wc-refunded', 'wc-failed'];

        // Locale d'origine pour restauration
        $orig_locale  = \get_locale();

        try {
            // Parcourt chaque langue/locale
            foreach ( $langs as $i => $lang ) {
                $locale = isset($locales[$i]) ? $locales[$i] : $orig_locale;

                if ( $locale !== $orig_locale ) {
                    // 1) Bascule le locale WordPress (gettext)
                    if ( function_exists('switch_to_locale') ) {
                        \switch_to_locale( $locale );
                    }
                    // 2) Décharge le textdomain WooCommerce (on se base sur ce qui "fonctionne" chez toi)
                    \unload_textdomain( 'woocommerce' );
                }

                foreach ( $wc_statuses as $status => $status_name_default ) {
                    if ( ! empty( $args['idStatus'] ) && $status !== $args['idStatus'] ) {
                        continue;
                    }
                    $deleted = in_array( $status, $deleted_map, true ) ? 1 : 0;

                    // 3) Récupère les libellés localisés dans la locale courante
                    $localized_statuses = \wc_get_order_statuses();
                    $label = isset($localized_statuses[$status]) ? $localized_statuses[$status] : $status_name_default;

                    $ordersStatus[] = [
                        'id_status' => $status,
                        'name'      => $label,
                        'lang'      => $lang,
                        'deleted'   => $deleted,
                    ];
                }

                if ( $locale !== $orig_locale ) {
                    // Restaure la locale d'origine (et nettoie le domain)
                    if ( function_exists('switch_to_locale') ) {
                        \switch_to_locale( $orig_locale );
                    }
                    if ( function_exists('unload_textdomain') ) {
                        \unload_textdomain('woocommerce');
                    }
                }
            }
        } catch (\Throwable $e) {
            // Fallback minimal en locale courante
            $ordersStatus = [];
            foreach ( $wc_statuses as $status => $status_name_default ) {
                if ( ! empty( $args['idStatus'] ) && $status !== $args['idStatus'] ) {
                    continue;
                }
                $ordersStatus[] = [
                    'id_status' => $status,
                    'name'      => $status_name_default,
                    'lang'      => substr(\get_locale(), 0, 2),
                    'deleted'   => in_array($status, $deleted_map, true) ? 1 : 0,
                ];
            }
        }

        // Pagination après expansion multilingue
        if ( isset($args['start'], $args['limit']) ) {
            $ordersStatus = array_slice( $ordersStatus, (int)$args['start'], (int)$args['limit'] );
        }

        $lastUpdate = date('Y-m-d H:i:s');
        return \apply_filters( __METHOD__, compact('lastUpdate','ordersStatus'), $args );
    }

    /**
     * Renvoie une liste des commandes correspondantes aux paramètres d'entrée
     * @param	timestamp 	$start
     * @param	integer 	$limit
     * @param	timestamp 	$lastUpdate
     * @param	integer		$idOrder
     * @param	bool 		$justCount
     * @return 	array 		( lastUpdate => timestamp, orders => array( shop_id_shop, order_is_confirm, order_reference, id_cart, id_carrier, id_address_delivery, id_address_invoice, id_status, date_cart, id_order, lang, amount, amount_without_tax, shipping, shipping_without_tax, tax_rate, currency_rate, currency, date_order, date_update, voucher_used, voucher_amount, products, customer, shipping_number, custom, ) )
     */
    public function syncOrders( $args ) {
        $order_args = array(
            'type'		=> 'shop_order',
            'limit'     => -1,
        );

        if (version_compare(WC()->version, '4.2.0', '>=')){
            $order_args['orderby'] = array(
                'modified' => 'ASC',
                'ID' => 'ASC'
            );
        }else {
            $order_args['orderby'] = 'ID';
            $order_args['order'] = 'ASC';
        }

        $orders = array();
        $justCount = isset($args['justCount']) && $args['justCount'];

        if (isset($args['lastUpdate']) && $args['lastUpdate'] && $args['lastUpdate'] != "") {
            $order_args['date_modified'] = ">=" . strtotime(get_gmt_from_date($args['lastUpdate']));
        }
        if (isset($args['limit']) && intval($args['limit'])) {
            $order_args['limit'] = $args['limit'];
        }
        if (isset($args['start']) && $args['start'] ) {
            $order_args['offset'] = $args['start'];
        }

        if ($justCount) {
            $order_args['limit'] = 1;
            $order_args['paginate'] = true;
        }

        $blog_ids = $this->getBlogIds( $args );

        foreach($blog_ids as $blog_id ) {
            if (function_exists( 'switch_to_blog' ) && \get_current_blog_id() != $blog_id) {
                \switch_to_blog( $blog_id );
            }

            if (isset($args['idOrder']) && intval($args['idOrder'])) {
                $wc_orders = array(\wc_get_order( $args['idOrder'] ));
            } else {
                $wc_orders = \wc_get_orders( $order_args );
            }

            if ($justCount) {
                $orders = array('count' => isset($wc_orders->total) ? $wc_orders->total : 0);
            } else if ($wc_orders && is_array($wc_orders)) {
                foreach ($wc_orders as $WC_Order) {
                    /* Formater la commande */
                    try {
                        $spm_order = $this->getOrder($WC_Order);
                    } catch(Exception $e) {
                        $spm_order = false;
                    }
                    if ($spm_order) {
                        $orders[] = $spm_order['order'];
                    }
                }
            }
        }
        if (function_exists('restore_current_blog')) {
            \restore_current_blog();
        }

        $lastUpdate = date('Y-m-d H:i:s');
        $return = compact('lastUpdate', 'orders');
        return \apply_filters( __METHOD__,  $return, $args);
    }

    /**
     * Renvoie les informations détaillées d'une commande
     * @param 	object 	$WC_Order
     * @return 	array|bool 		( order => array( shop_id_shop, order_is_confirm, order_reference, id_cart, id_carrier, id_address_delivery, id_address_invoice, id_status, date_cart, id_order, lang, amount, amount_without_tax, shipping, shipping_without_tax, tax_rate, currency_rate, currency, date_order, date_update, voucher_used, voucher_amount, products, customer, shipping_number, custom ) )
     */
    public function getOrder( $WC_Order ) {
        $idOrder = (is_object($WC_Order) && method_exists($WC_Order,'get_id')) ? $WC_Order->get_id() : false;
        if (!$idOrder) {
            return false;
        }

        /* Ne pas prendre en compte les commandes des administrateurs */
        $WC_Customer = false;
        if ($WC_Order->get_user_id() > 0) {
            $WC_Customer = new \WC_Customer($WC_Order->get_user_id());
            if (!$WC_Customer) {
                return false;
            } else if (in_array('administrator', array($WC_Customer->get_role()))){
                return false;
            }
        }

        // 🔁 Langue de la commande
        if (function_exists('pll_get_post_language')) {
            $lang = pll_get_post_language($idOrder);
        }
        if (empty($lang)) {
            $lang = substr(get_locale(), 0, 2);
        }

        $order = array(
            'shop_id_shop'			=> $this->getShopID(),
            'order_is_confirm'		=> 0,
            'order_reference'		=> $idOrder,
            'id_cart'				=> '',
            'id_carrier'			=> 0,
            'id_address_delivery'	=> getAddress($WC_Order->get_address('shipping'))['id_address'],
            'id_address_invoice'	=> getAddress($WC_Order->get_address('billing'))['id_address'],
            'id_status'				=> 'wc-' . $WC_Order->get_status(),
            'date_cart'				=> $WC_Order->get_date_created()->date_i18n( 'Y-m-d H:i:s' ),
            'id_order'				=> $idOrder,
            'lang'					=> $lang,
            'amount'				=> floatval($WC_Order->get_total()),
            'amount_without_tax'	=> floatval($WC_Order->get_total() - $WC_Order->get_total_tax()),
            'shipping'				=> floatval($WC_Order->get_shipping_total()),
            'shipping_without_tax'	=> floatval(floatval($WC_Order->get_shipping_total()) - floatval($WC_Order->get_shipping_tax())),
            'tax_rate'				=> 1,
            'currency_rate'			=> 1,
            'currency'				=> $WC_Order->get_currency(),
            'date_order'			=> $WC_Order->get_date_created()->date_i18n( 'Y-m-d H:i:s' ),
            'date_update'			=> $WC_Order->get_date_modified() ?
                $WC_Order->get_date_modified()->date_i18n( 'Y-m-d H:i:s' ) :
                false,
            'voucher_used'			=> (version_compare(WC()->version, '3.7.0', '<') ? $WC_Order->get_used_coupons():$WC_Order->get_coupon_codes()),
            'voucher_amount'		=> $WC_Order->get_discount_total(),
            'products'				=> getOrderProducts($WC_Order->get_items()),
            'customer'				=> ($WC_Customer) ?
                $this->formatUser(array('WC_Customer' => $WC_Customer)) :
                $this->formatOrderGuest(array('shopIdShop' => $this->getShopID(), 'WC_Order' => $WC_Order)),
            'shipping_number'		=> '',
            'custom' 				=> array(),
        );

        if ($WC_Order->get_date_paid() &&
            !in_array($WC_Order->get_status(), array('refunded', 'canceled'))) {
            $order['order_is_confirm'] = 1;
        }

        if (metadata_exists('post', $idOrder, 'spmid_cart_order')) {
            $order['id_cart'] = get_post_meta($idOrder, 'spmid_cart_order', true);
        } else {
            /* Récupérer l'id_cart */
            $id_last_cart_user = get_user_meta($WC_Order->get_user_id(), 'spmid_last_cart', true);
            /* Lié l'id cart à la commande */
            if ($id_last_cart_user && $id_last_cart_user != "") {
                add_post_meta($idOrder, 'spmid_cart_order', $id_last_cart_user, true);
                $order['id_cart'] = $id_last_cart_user;
                delete_user_meta($WC_Order->get_user_id(), 'spmid_cart_order');
            }
        }

        $shipping_methods = $WC_Order->get_shipping_methods();
        if ($shipping_methods) {
            $shipping_method = array_pop( $shipping_methods );
            try {
                $shipping_method_data = $shipping_method->get_data();
            } catch(Exception $e) {
                $shipping_method_data = array();
            }

            $shipping_method_instance_id = array_key_exists('instance_id', $shipping_method_data) ? $shipping_method_data['instance_id'] : 0;

            /* Si la clé instance_id n'est pas définit,
                On récupére instance_id depuis method_id */
            if (!$shipping_method_instance_id && array_key_exists('method_id', $shipping_method_data)) {
                $shipping_method_id = $shipping_method_data['method_id'];
                if (strpos($shipping_method_id, ':')) {
                    $explode_shipping_method_id = explode(":", $shipping_method_id);
                    $shipping_method_instance_id = $explode_shipping_method_id[1];
                }
            }

            $order['id_carrier'] = intval($shipping_method_instance_id);
        }

        $return = compact( 'order' );
        return \apply_filters( __METHOD__,  $return);
    }


    /**
     * Renvoie les données détaillées d'un invité ayant passé une commande
     * @param 	array $args
     * @return 	array 	( user => array( id_customer, shop_id_shop, locale, first_name, last_name, birthday, email, optin, newsletter, customer_since, customer_update, gender, groups, active, addresses, custom ) )
     */
    public function formatOrderGuest( $args ) {
        $WC_Order = isset( $args['WC_Order'] ) ? $args['WC_Order'] : false;
        $shopIdShop = isset( $args['shopIdShop'] ) ? $args['shopIdShop'] : false;

        $guest = array();

        if ($WC_Order && $shopIdShop) {
            $email_guest = $WC_Order->get_billing_email();

            $guest = array(
                'id_customer'	    => $email_guest,
                'shop_id_shop'	    => $shopIdShop,
                'locale'		    => \get_locale(),
                'first_name'	    => $WC_Order->get_billing_first_name(),
                'last_name'		    => $WC_Order->get_billing_last_name(),
                'birthday'		    => '0000-00-00',
                'email'			    => $email_guest,
                'optin'			    => 0,
                'newsletter'	    => 0,
                'customer_since'	=> '',
                'customer_update'	=> '',
                'gender'		    => 0,
                'groups'		    => array(),
                'active'		    => 1,
                'addresses'		    => array(),
                'nb_order' 		    => 0,
                'sum_order' 	    => 0,
                'nb_order_year'     => 0,
                'sum_order_year'    => 0,
                'custom'		    => array(),
                'date_last_order'	=> null,
            );

            $billing_address = $WC_Order->get_address();
            if (implode('', $billing_address ) !== '') {
                $formattedBillingAddress = getAddress( $billing_address );
                $guest['addresses'][] = $formattedBillingAddress;
            }
        }

        return \apply_filters( __METHOD__,  $guest, $args );
    }


    /** END COMMANDES **/

    /** MAILING LIST **/

    /**
     * Renvoie une liste détaillée d'utilisateurs ayant souscrit à la newsletter
     * Permet de récupérer la liste des contacts abonnés NL présents dans la base
     * Cette méthode est utilisée pour la fonctionnalité de campagne SMS et Email
     * @param	timestamp 	$start
     * @param	integer 	$limit
     * @param	timestamp 	$lastUpdate
     * @param	bool 		$justCount
     * @return 	array 		( lastUpdate => timestamp, newsletters => array( id_shop, last_name, first_name, email, zipcode, last_update, newsletter ) )
     */

    public function syncNewsletters( $args ) {
        $newsletters = array();

        /* Cas autre
        $newsletter_field = \ShopiMindConfiguration::getUserNewsletterField();
        echo " field = "  .$newsletter_field;
        if( $newsletter_field != '-1' ) {
        }*/


        $fields_translation = \ShopiMindConfiguration::getMLSyncFields();

        $newsletter_table = \ShopiMindConfiguration::getMLTableName();
        if( $newsletter_table && $newsletter_table != 'none' ) {
            global $wpdb;

            $sql = "SELECT ";
            $keys = array();
            if( isset( $args['justCount'] ) && $args['justCount'] ) {
                $sql .= " COUNT(*) AS `count` ";
            }else {
                foreach( $fields_translation as $spm_field => $table_field ) {
                    if( $table_field === '-' ) {
                        continue;
                    }
                    $keys[] = " $table_field as $spm_field ";
                }
                $sql .= implode(', ', $keys );
            }

            $sql .= " FROM $newsletter_table ";


            if(isset($args['lastUpdate'] )  && $args['lastUpdate'] && array_key_exists('last_update', $fields_translation) && $fields_translation['last_update'] !== '-') {
                $sql .= " WHERE `".$fields_translation['last_update']."` >= ".'"'.esc_sql($args['lastUpdate']).'"';
            }
            if((!isset( $args['justCount'] ) || !$args['justCount']) && array_key_exists('last_update', $fields_translation) && $fields_translation['last_update'] !== '-') {
                $sql .= " ORDER BY `".$fields_translation['last_update']."`";

            }
            if((!isset( $args['justCount'] ) || !$args['justCount']) && isset($args['limit'] ) ) {
                $sql .= " LIMIT ".(int)$args['limit'].(isset ($args['start'] ) ?" OFFSET ".(int)$args['start']:"");
            }
            $newsletters = $wpdb->get_results( $sql, ARRAY_A );

            if( isset( $args['justCount'] ) && $args['justCount'] ) {
                if(is_array($newsletters) && isset($newsletters[0]) && is_array($newsletters[0]) && isset($newsletters[0]['count'])) {
                    $newsletters = array( 'count' => $newsletters[0]['count'] );
                }else {
                    $newsletters = array( 'count' => 0 );
                }
            }else {
                foreach( $newsletters as $i => $newsletter ) {

                    // plugin : https://www.thenewsletterplugin.com/
                    if( $newsletter['newsletter'] == 'C' ) {
                        $newsletters[$i]['newsletter'] = 1;
                    }
                    elseif( $newsletter['newsletter'] == 'U' ) {
                        $newsletters[$i]['newsletter'] = 0;
                    }

                }
            }
        }
        else {
            return $newsletters;
        }
        $lastUpdate = date('Y-m-d H:i:s');
        $return = compact( 'lastUpdate', 'newsletters' );
        return \apply_filters( __METHOD__,  $return, $args );
    }

    /**
     * Désinscription d'un customer de la newsletter
     * @param 	integer $idCustomer
     * @param 	string 	$email
     * @return 	array 	( success => bool )
     */
    public function unsubCustomer( $args ) {
        // $newsletter_field = \ShopiMindConfiguration::getUserNewsletterField();

        return array(
            'success'	=> true
        );
    }

    /** END MAILING LIST **/

    /*** fonction sinternes ***/

    /**
     * Renvoie une liste multilingue détaillée d’un produit
     * @param	WC_Product	$WC_Product
     * @return 	array 		Tableau indexé contenant une version formatée du produit par langue (avec id_product de la langue par défaut)
     */

    public function formatProduct($WC_Product) {
        // Si Polylang n'est pas actif, retourne le produit tel quel dans un tableau
        if (! function_exists('pll_get_post_translations')) {
            $product = $this->formatSingleProduct($WC_Product);
            return [ $product ];
        }

        $translations = pll_get_post_translations($WC_Product->get_id());
        $default_lang = pll_default_language();
        $default_product_id = isset($translations[$default_lang]) ? intval($translations[$default_lang]) : $WC_Product->get_id();
        $result = [];

        foreach ($translations as $lang => $post_id) {
            $translated_product = wc_get_product($post_id);
            if (! $translated_product) {
                continue;
            }

            $product_data = $this->formatSingleProduct($translated_product, $lang);

            // id_product forcé sur l’ID par défaut pour toutes les langues
            $product_data['id_product'] = $default_product_id;
            $product_data['lang'] = $lang;

            $result[] = $product_data;
        }

        return $result;
    }


    /**
     * Renvoie une version détaillée d’un produit dans une langue donnée
     * @param	WC_Product	$WC_Product
     * @param	string|null	$lang	Code langue (ex. 'fr'), ou null pour la langue courante
     * @return 	array 		Tableau contenant les données du produit dans la langue spécifiée
     */

    protected function formatSingleProduct($WC_Product, $lang = null) {
        if ($lang && function_exists('pll_set_language')) {
            pll_set_language($lang);
        }

        $price = $WC_Product->get_regular_price();
        $priceDiscount = $WC_Product->get_sale_price() > 0 ? $WC_Product->get_sale_price() : $WC_Product->get_regular_price();

        if (get_option('woocommerce_prices_include_tax') == 'no' && get_option('woocommerce_tax_display_shop') == 'incl') {
            $price = wc_get_price_including_tax($WC_Product, array('price' => $price));
            $priceDiscount = wc_get_price_including_tax($WC_Product, array('price' => $priceDiscount));
        }

        $active = $WC_Product->get_status() == 'publish' ? 1 : 0;

        $product = array(
            'shop_id_shop'       => $this->getShopID(),
            'id_product'         => intval($WC_Product->get_id()), // sera écrasé par la version multilangue
            'reference'          => $WC_Product->get_sku(),
            'lang'               => substr(get_locale(), 0, 2), // sera remplacé si multilingue
            'name'               => $WC_Product->get_name(),
            'description_short'  => $WC_Product->get_short_description(),
            'description'        => $WC_Product->get_description(),
            'product_link'       => $WC_Product->get_permalink(),
            'image_link'         => '',
            'images'             => array(),
            'id_categories'      => array(),
            'id_manufacturer'    => 0,
            'currency'           => get_woocommerce_currency(),
            'price'              => floatval($price),
            'price_discount'     => floatval($priceDiscount),
            'quantity_remaining' => false,
            'date_creation'      => $WC_Product->get_date_created()->date_i18n('Y-m-d H:i:s'),
            'date_update'        => $WC_Product->get_date_modified()->date_i18n('Y-m-d H:i:s'),
            'custom'             => array(),
            'combinations'       => array(),
        );

        $default_variation = $WC_Product->get_default_attributes();

        if ($WC_Product->get_manage_stock()) {
            $product['quantity_remaining'] = floatval($WC_Product->get_stock_quantity());
        }

        if ($active && \ShopiMindConfiguration::excludeOutOfStockProducts() && !$WC_Product->is_in_stock()) {
            $active = 0;
        }

        $product['active'] = $active;

        $product['id_categories'] = wp_get_post_terms($WC_Product->get_id(), 'product_cat', array('fields' => 'ids'));
        $product['id_categories'] = array_map('intval', $product['id_categories']);

        $manufacturer_field = \ShopiMindConfiguration::getProductManufacturerField();
        if ($manufacturer_field != 'none') {
            $product['id_manufacturer'] = $WC_Product->get_meta($manufacturer_field);
        }

        $product['image_link'] = wp_get_attachment_image_src($WC_Product->get_image_id(), \ShopiMindConfiguration::getEmailImageFormat())[0];
        $attachments_id = $WC_Product->get_gallery_image_ids();
        if ($attachments_id) {
            foreach ($attachments_id as $attachment_id) {
                $image = array(
                    'id_image'            => $attachment_id,
                    'id_product'          => $WC_Product->get_id(),
                    'id_product_attribute'=> null,
                    'url'                 => wp_get_attachment_image_src($attachment_id, \ShopiMindConfiguration::getEmailImageFormat())[0],
                    'default'             => false,
                );
                $product['images'][] = $image;
            }
        }

        if ($WC_Product->is_type('variable')) {
            $wc_products_variation = $WC_Product->get_available_variations();
            $defaultVariationPrice = false;

            if (is_array($wc_products_variation) && count($wc_products_variation) > 0) {
                foreach ($wc_products_variation as $WC_Product_Variation) {
                    $image_variation = $WC_Product_Variation['image']['url'];
                    $WC_Product_Variation = wc_get_product($WC_Product_Variation['variation_id']);

                    $price = $WC_Product_Variation->get_regular_price();
                    $priceDiscount = $WC_Product_Variation->get_sale_price() > 0 ? $WC_Product_Variation->get_sale_price() : $WC_Product_Variation->get_regular_price();

                    if (get_option('woocommerce_prices_include_tax') == 'no' && get_option('woocommerce_tax_display_shop') == 'incl') {
                        $price = wc_get_price_including_tax($WC_Product_Variation, array('price' => $price));
                        $priceDiscount = wc_get_price_including_tax($WC_Product_Variation, array('price' => $priceDiscount));
                    }

                    $array_p = array(
                        'id_combination'     => intval($WC_Product_Variation->get_id()),
                        'combination_name'   => $WC_Product_Variation->get_name(),
                        'reference'          => $WC_Product_Variation->get_sku(),
                        'product_link'       => $WC_Product_Variation->get_permalink(),
                        'images'             => $product['images'],
                        'image_link'         => $image_variation,
                        'price'              => floatval($price),
                        'price_discount'     => floatval($priceDiscount),
                        'quantity_remaining' => 0,
                        'default'            => 0,
                        'values'             => array(),
                    );

                    if ($WC_Product_Variation->get_manage_stock()) {
                        $array_p['quantity_remaining'] = floatval($WC_Product_Variation->get_stock_quantity());
                    }

                    $product_attributes = get_post_meta($WC_Product->get_id(), '_product_attributes', true);
                    $attributes = $WC_Product_Variation->get_attributes();
                    if ($attributes) {
                        $matching_default_attributes = array();
                        foreach ($attributes as $attribute => $value) {
                            $attribute_name = isset($product_attributes[$attribute]) ? $product_attributes[$attribute]['name'] : $attribute;
                            $array_p['values'][] = array(
                                'name'  => $attribute_name,
                                'value' => $value,
                            );
                            $matching_default_attributes[$attribute_name] = $value;
                        }
                    }

                    if ($matching_default_attributes == $default_variation) {
                        $array_p['default'] = true;
                    }

                    if (!$defaultVariationPrice) {
                        $parent_price = $array_p['price'];
                        $parent_price_discount = ($array_p['price_discount'] > 0) ? $array_p['price_discount'] : $array_p['price'];
                        if ($array_p['default']) {
                            $product['price'] = $parent_price;
                            $product['price_discount'] = $parent_price_discount;
                            $defaultVariationPrice = true;
                        } else {
                            if ($product['price'] == 0 || ($product['price'] > $array_p['price'])) {
                                $product['price'] = $parent_price;
                                $product['price_discount'] = $parent_price_discount;
                            }
                        }
                    }

                    $product['combinations'][intval($WC_Product_Variation->get_id())] = $array_p;
                }
            }
        }

        return $product;
    }


    /**
     * Mise a jour du script url en bas de page
     * @param 	array 	script_url
     * @return 	array 	( success => bool )
     */
    public function setScriptUrl( $args ) {
        $new_script_url = $args['script_url'];

        if(!$new_script_url || $new_script_url == '' || $new_script_url == 'none') {
            $new_script_url = 'https://app-spm.com';
        }
        update_option('spm_script_url', $new_script_url);

        $return = array(
            'success'	=> true
        );
        return \apply_filters( __METHOD__,  $return, $args );
    }


}