<?php
namespace ShopiMind;

if( !class_exists( 'APIError' )) {
	require_once( trailingslashit( dirname( __FILE__ ) ) . 'error.class.php' );
}

require_once( trailingslashit( dirname( __FILE__ ) ) . 'api.functions.php' );

class APIServer {
	protected $_apiVersion = '1';
	protected $_apiLastModification = '20190213';

	public $method 	= '';
	public $query_vars = array();
	public $allowed_parameters = array();

	public $blog_id = false;

	// allowed methods
	protected $_methods = array();
	// parameters validation
	protected $_parameters_validation = array();
	protected $_mandatory_params = array();

	function __construct( $request = false ) {
		$this->request = $request;
		$this->blog_id = \get_current_blog_id();
	}

	public function getModuleVersion() {
		return SHOPIMIND_PLUGIN_VERSION;
	}

    /**
     * Permet de récupérer la build du client
     *
     * @return int
     */
    public function getBuild() {
        return 128;
    }

	protected function validateField( $key, $value ) {
		if( empty( $key ))
			return new APIError( 'parameter', sprintf( __( "Champ inconnu pour %s", 'shopimind' ), $value ));

		if( empty( $this->_parameters_validation ) || !isset( $this->_parameters_validation[$key] ))
			return new APIError( 'parameter', sprintf( __( "Pas de méthode de validation pour '%s' (valeur = %s)", 'shopimind' ), $key, $value ));

		$filter = $this->_parameters_validation[$key];
		if( $filter === true )
			return $value;
		elseif( $filter === 'string' ) {
			return filter_var( $value, FILTER_SANITIZE_STRING );
		}
		elseif( $filter === FILTER_VALIDATE_INT ) {
			return filter_var( $value, FILTER_VALIDATE_INT );
		}
		elseif( $filter == 'timestamp' ) {
			if ( strlen($value) && $value != 0 && !\DateTime::createFromFormat( 'Y-m-d H:i:s', $value ) ) {
				return new APIError( 'parameter', sprintf( __( "Format de date non valide pour le paramètre '%s' (valeur = %s)", 'shopimind' ), $key, $value ));
			}
			else {
				return $value;
			}
		}
		elseif( is_int( $filter )) {
			return filter_var( $value, $filter );
		}
		else {
			die( "NO FILTER FOR $key = $value" );
		}
	}

	function setMethod( $request_method ) {
		if( !strlen( $request_method )) {
			return new APIError( 'method', sprintf( __( "Aucune méthode indiquée (Erreur %d)", 'shopimind' ), 1 ) );
		}

		if( !isset( $this->_methods[$request_method] )) {
			return new APIError( 'method', sprintf( __( "Méthode inconnue : '%s'", 'shopimind' ), $request_method ) );
		}

		$this->method = $request_method;
		$this->allowed_parameters = $this->_methods[$request_method];
		return $this->method;

	}

	function parseRequest() {
		foreach( $this->allowed_parameters as $allowed_parameter ) {
			if( array_key_exists( $allowed_parameter, $this->request )) {
				$value = $this->validateField( $allowed_parameter, $this->request[$allowed_parameter] );
				if( is_api_error( $value )) {
					return $value;
				}
				if( $value ) {
					$this->query_vars[$allowed_parameter] = $value;
				}
			}
		}
	}

	function canExecute() {
		if( empty( $this->method )) {
			return new APIError( 'method', sprintf( __( "Aucune méthode indiquée (Erreur %d)", 'shopimind' ), 2 ) );
		}
		if( !method_exists( $this, $this->method )) {
			return new APIError( 'method', sprintf( __( "Méthode inconnue : '%s'", 'shopimind' ), $this->method ) );
		}

		if( isset( $this->_mandatory_params[$this->method] )) {
			foreach( $this->_mandatory_params[$this->method] as $mandatory_param ) {
				if( !isset( $this->query_vars[$mandatory_param] )) {
					$string = 'query = ' . print_r($this->query_vars, 1) . " request = " . print_r($this->request, 1);
					return new APIError( 'parameter', sprintf( __( "Paramètre '%s' obligatoire manquant $string", 'shopimind' ), $mandatory_param ) );
				}
			}
		}
		return true;
	}

	function execute() {
		return $this->{$this->method}( $this->query_vars );
	}

	function sendJSON( $array ) {
		wp_send_json( $array );
	}

	function sendError( $message = '' ) {
		$success = false;
		$message = filter_var( $message, FILTER_SANITIZE_STRING );

		if($message) {
			$return = compact( 'success', 'message' );
		}
		else {
			$return = compact( 'success' );
		}

		$this->sendJSON( $return );
	}

	function sendSuccess( $data ) {
		$success = array( 'success'	=> true );
		if( $data === true ) {
			$data = $success;
		}
		else {
			$data = array_merge( $success, $data );
		}

		$this->sendJSON( $data );
	}
}
